//---------------------------------------------------------------------------
//
// %FILE     uart.c
// %VSS-REV  $Revision: 14 $
// %CREATED  1996.03.01
// %REVISED  $Date: 4/18/97 4:25p $
// %AUTHOR   Michael C. Draeger 
// %PROJECT  NS486SXF evaluation board software
// %PART     NS486SXF (B0+), NS486SXL
// %SUMMARY  console code for debug UART
//     
// %VSS      $Author: Miked $ $Date: 4/18/97 4:25p $ $Revision: 14 $
//
// DESCRIPTION
//
//   Code to initialize, print to, and read from the internal UART.
//   This code uses polling (non-interrupt) mode.
//
// INPUTS
//
// HISTORY
//
/*
 *
 * $History: uart.c $ 
 * 
 * *****************  Version 14  *****************
 * User: Miked        Date: 4/18/97    Time: 4:25p
 * Updated in $/nsdemo
 * Added SXL support.   New header (comment) changes.
 * 
 * *****************  Version 13  *****************
 * User: Miked        Date: 4/07/97    Time: 1:27p
 * Updated in $/nsdemo
 * Updated comments for SXL release.
 * 
 * *****************  Version 12  *****************
 * User: Miked        Date: 12/04/96   Time: 4:04p
 * Updated in $/nsdemo
 * Added typecast to clarify code.
 * 
 * *****************  Version 11  *****************
 * User: Miked        Date: 12/04/96   Time: 1:41p
 * Updated in $/nsdemo
 * Changed version to 1.6
 * 
 * *****************  Version 10  *****************
 * User: Miked        Date: 10/03/96   Time: 4:55p
 * Updated in $/nsdemo
 * Added function to read character (polled) from UART.
 * 
 * *****************  Version 9  *****************
 * User: Miked        Date: 8/06/96    Time: 11:59a
 * Updated in $/nsdemo
 * Version 1.4.  Maintainance release.  See README.TXT for info.
 * 
 * *****************  Version 8  *****************
 * User: Miked        Date: 7/23/96    Time: 2:26p
 * Updated in $/nsdemo
 * Maintainance release.  README.TXT describes changes.
 * 
 * *****************  Version 7  *****************
 * User: Miked        Date: 7/16/96    Time: 11:54a
 * Updated in $/nsdemo
 * Updated for rev C0 release.
 * 
 * *****************  Version 6  *****************
 * User: Miked        Date: 5/03/96    Time: 2:50p
 * Updated in $/nsdemo
 * Maintainence release.
 * 
 * *****************  Version 5  *****************
 * User: Miked        Date: 4/18/96    Time: 10:52a
 * Updated in $/nsdemo
 * Clean up for release.
 * 
 * *****************  Version 4  *****************
 * User: Miked        Date: 4/12/96    Time: 11:54a
 * Updated in $/nstest
 * Uses DeviceInfo and TargetInfo to do UART bug patch and set clock rate
 * divisor.
 * 
 * *****************  Version 3  *****************
 * User: Miked        Date: 4/09/96    Time: 11:06a
 * Updated in $/board test
 * Added code so that clock divisor is set up based on CPU speed.  Changed
 * to VSS headers.  Added uart.h.
 *
 */
//
// COPYRIGHT
//
//      (c) 1996, 1997 National Semiconductor Corporation
//
//---------------------------------------------------------------------------

#include "uart.h"

//---------------------------------------------------------------------------

USHORT UART_Initialize()
{

  BYTE tByte;

  // If UART is not enabled in the BIU, return FAIL

  if ( ( IOR_BYTE( BIU_CONTROL1 ) & 0x40 ) != 0x40 )
    return FAIL;

  // NS486SXF Rev B UART/RIO bug

  if ( ( gDeviceInfo.id == ID_NS486SXF ) 
    && ( gDeviceInfo.rev == SXF_REV_B ) )
  {
    tByte = IOR_BYTE ( RIO_CONTROL );
    tByte |= 0x10;                        // set NoUART bit
    IOW_BYTE ( RIO_CONTROL, tByte );

    tByte = IOR_BYTE ( RIO_DD_BYTE0 );
    tByte &= 0xCF;                        // set direction to input for UART
    IOW_BYTE ( RIO_DD_BYTE0, tByte );
  }

  // setup Clock divisor based on CPU speed

  if ( gTargetInfo.valid == TRUE )
  {  
    switch( gTargetInfo.cpu_speed )
    {
    case 25:
      IOW_BYTE ( UART_CLOCK, 27 );    // 25MHz          
      break;
    case 20:
      IOW_BYTE ( UART_CLOCK, 22 );    // 20MHz      
      break;
    default:
      IOW_BYTE ( UART_CLOCK, 27 );    // 25MHz      
      break;
    }
  }
  else
    IOW_BYTE( UART_CLOCK, 27 );       // 25MHz
  
  // set up UART pins

  IOW_BYTE ( UART_MSC, 0    );          // not IR, dont need extra pins

  // Initialize UART

  IOW_BYTE( UART_LCR, 0x03  );    // clear DLAB, no parity, 1 stop, 8 bit word 
  IOW_BYTE( UART_IER, 0x00  );    // disable all interrupts (use polling)
  IOW_BYTE( UART_FCR, 0x00  );    // disable FIFOs (if 16550)
  IOW_BYTE( UART_MCR, 0x03  );    // set DTR and RTS
  IOW_BYTE( UART_LCR, 0x83  );    // set DLAB, no parity, 1 stop, 8 bit word
  IOW_BYTE( UART_DLL, 12    );    // set baud rate LSB
  IOW_BYTE( UART_DLM, 0     );    // set baud rate MSB
  IOW_BYTE( UART_LCR, 0x03  );    // clear DLAB, no parity, 1 stop, 8 bit word

  return SUCCESS;

}

//---------------------------------------------------------------------------

// limited to 200 characters right now
void uart_printf( const char *fmt, ... )
{
	
  va_list arg;
  char buf[200];

  va_start (arg, fmt);
  vsprintf( buf, (char *)fmt, arg);
  uart_outs( buf );        
  va_end (arg);

}

//---------------------------------------------------------------------------

USHORT uart_outs( char * str )
{

  int position = 0;

  while ( str[position] != '\0' )
  {
    uart_outc( str[position] );
    position++;
  }

  return SUCCESS;

}

//---------------------------------------------------------------------------

USHORT uart_outc( char ch )
{
  
  int byteval;  
  BOOL loopflag = TRUE;

  while ( loopflag )
  {
    byteval = IOR_BYTE ( UART_LSR );    // read line status to see if uart is busy
    if ( (byteval & 0x20)==0x20 )       // THRE bit
      loopflag = FALSE;
  }
  IOW_BYTE ( UART_THR, ch );        // write byte

  return SUCCESS;

}

//---------------------------------------------------------------------------

char uart_getc()
{

  int byteval;  
  BOOL loopflag = TRUE;

  while ( loopflag )
  {
    byteval = IOR_BYTE ( UART_LSR );    // read line status to see if uart is busy
    if ( (byteval & 0x01)==0x01 )       // THRE bit
      loopflag = FALSE;
  }
  return (char) IOR_BYTE ( UART_RBR );        // write byte

}

//---------------------------------------------------------------------------
// END       uart.c
//---------------------------------------------------------------------------
