/***********************************************************************
 *
 *	File name:  	linkCom.c
 *
 *	Description:	This file contains functions used by the
 * 			CHAMELEAN to initiate the LINK layer and
 *			to set up its ports for BASIC RATE ACCESS.
 *
 *	Functions:	Basic Rate macro definitions
 *                      reset_port()
 *                      setup()
 *			s_lnkstat()
 *			s_tran()
 *                      TEI_req()
 *                      TEI_ASS()
 *                      setupLinks()
 *
 *      Version		Date		ID	     Comment
 *	________________________________________________________________
 *	0.1		110488		TEKELEC/IA   First created
 *      1.0             112188          TEKELEC/IA   First release
 *      1.2             120288          TEKELEC/IA   r[5], c[5] moved 
 *      1.3             120688          TEKELEC/IA   extern rxlen removed 
 *      1.4             120888          TEKELEC/IA   Demo package
 *
 ***********************************************************************/

#include	"mainSym.h"




/*
 *   MACRO DEFINITIONS FOR BASIC RATE LIBRARY.
 *
 *
 */


#define		SB		SetBasic(c,r)
#define		reset_basic(x)	c[0]=100*x + 3; SB
#define		set_MODE(x,y)   c[0]=100*x + 1; c[1]=y; SB
#define		set_dch(x,m)	c[0]=100*x + 4; c[1]=1; c[2]=3; c[3]=m; SB
#define		set_b1ch(x,m)   c[0]=100*x + 4; c[1]=1; c[2]=1; c[3]=m; SB
#define		set_b2ch(x,m)   c[0]=100*x + 4; c[1]=1; c[2]=2; c[3]=m; SB
#define		set_bch(y,x,m)	c[0]=100*x + 4; c[1]=1; c[2]=y; c[3]=m; SB
#define		set_ntp(x,y)	c[0]=100*x + 10;c[1]=y; SB
#define     	get_status(x)   c[0]=100*x + 6; SB



/*
 *	Function:	reset_port()
 *	Description:	Reset Basic Rate Interface 
 *
 *	Inparameters:   port number
 *
 *	Outparameters:  none
 *
 */

reset_port(portnum)
    {
    int r[5],c[5];
    reset_basic (portnum);
    if(r[0]!=0)  pr_error( (int) r[0],"reset_basic: ",STOP );
    } /* end reset_port */



/*
 *	Function:	setup()
 *	Description:	Initiation of specified port for
 *			Basic Rate Access. LINK layer timers
 *			packet size and retransmition numbers
 *			are set to system parameter values.
 *
 *	Inparameters:   port number, Chamelean mode, 
 *                      D-,B1- & B2-channel initial status
 *
 *	Outparameters:  none
 *                     
 *
 */


setup(portnum,mode,D_chan,B1_chan,B2_chan,ntp)
int    portnum,
       mode,
       D_chan,
       B1_chan,
       B2_chan,
       ntp;
    {
    long delay;
    int  r[5],c[5];

    /*
     *
     * Set Basic Rate Interface  mode 
     *
     */

    set_MODE(portnum,mode);
    if ( r[0] != 0) pr_error( (int) r[0],"set_MODE: ",STOP);


    /*
     * Set D_channel 
     */

    set_dch (portnum,D_chan);
    if (r[0] != 0) pr_error( (int) r[0],"set_dch: ",STOP);


    /*
     * Set B1_ and B2_channel 
     */

    set_b1ch(portnum,B1_chan);
    if (r[0] != 0) pr_error( (int) r[0],"set_b1ch  ",STOP);

    set_b2ch(portnum,B2_chan);
    if (r[0] != 0) pr_error( (int) r[0],"set_b2ch: ",STOP);

    /*
     *    Set which LAPD library to communicate with.
     */

    setport(portnum);
	

    /*
     *
     *    Set NT power.
     *
     */

    set_ntp(portnum,ntp); 
    if (r[0] != 0) pr_error( (int) r[0],"set_ntp: ",STOP);


    /*
     * Initiate communication processor
     */

    r[0] = initp1(INTERFACE,STATION,ENCODE,BITRATE);
    if (r[0]!= 0) pr_error( (int) r[0],"INITIP1: ",STOP);

    /*
     *  Set timers, packet size, retransmitions, window_size
     *  outstanding frames.
     */

    set_t1(T200);
    set_t2(T203); 

    set_n1(N201); 
    set_n2(N200); 

    set_window(WINDOW_SIZE); 
    set_mod(MODULUS); 

    set_sconfig(CONFIG); 

    }/*  end setup  */




/*
 *	Function:	s_lnkstat()
 *	Description:	This function waits for the link layer connection
 *			to be established. The status of layer is checked
 *                      before the link layer status is checked.
 *                      The user can abort the connection attempt
 *                      by pressing F10.
 *
 *	Inparameters:   PORT number
 *
 *	Outparameters:  none
 *			
 *
 */


s_lnkstat(portnum)
int	portnum;
    {
    int   state,c[5],r[5];
    get_status(portnum);
    if (r[0] != 0) pr_error( (int) r[0],"Layer 1 down: ", STOP);

    setport(portnum);

    while ( (state = status() ) != INFO_TRANSF )
	{
	pr_error(state,"s_lnkstat: ",CONT);
	if (getch(_stdvt) == F10) pr_error(portnum,
                                           "User abort: s_lnkstat:",STOP); 
	}

    } /* end s_lnkstat */





/*
 *	Function:	s_tran()
 *	Description:	This function transmits a layer 3 frame.
 *			The user can abort by pressing F10.
 *
 *	Inparameters:   port number
 *                      sapi index value
 *                      tei index value
 *			message type
 *			array containing information elements
 *			length of information element packet
 *
 *	Outparameters:  none
 *
 */

s_tran (portnum,tei,sapi,mestype,IE,ieleng,refLen,callRef)
byte	mestype,*IE,refLen,callRef;
int	portnum,tei,sapi;
    {
    byte  txbuffer[256],ch;
    int	  i,count,length,result;

    setport(portnum);
    set_tei(tei);
    set_sapi(sapi);

    /*	build layer 3 packet	*/

    txbuffer[0] = PD; 
    txbuffer[1] = refLen;
    
    i = 2 ;
    if( refLen == 0x01 ) txbuffer[i++] = callRef;  
    else if (refLen !=0)  pr_error( (int) refLen,
                                    "Call Ref length not supported: ",
                                    STOP);
    txbuffer[i++] = mestype;

    for(count = 0;count < ieleng;count ++)
        txbuffer[count+i] = IE[count];

    length = ieleng + i;	

    while( (result = transmit(txbuffer,length) ) != 0) 
        {
        pr_error(result,"s_trans: ",CONT);
	if (getch(_stdvt) == F10) pr_error(portnum,"User abort: s_tran",STOP);
	}

    }  /*  end s_tran  */




/*
 *	Function:	TEI_req()
 *	Description:	This function waits for an IDREQ UI frame.
 *                      The user can abort by pressing F10.
 *
 *	Inparameters:   UI frame
 *			port number
 *                      pointer to UI buffer
 *
 *	Outparameters:  UI buffer
 *
 */


TEI_req(portnum,ui)
int portnum;
byte *ui;
    {
    byte rxbuf[200],ch;    
    int  i,result;
    setport(portnum);
    rxlen = 0;
    rxbuf[4] = '\0';
    while( rxlen == 0 || rxbuf[4] != IDREQ) 
        {
        result = receive(rxbuf);
        if (result != 0 ) pr_error(result,"TEI_req: ",CONT);
	if ( getch(_stdvt) == F10 ) pr_error(portnum,
                                             "User abort: TEI_req",STOP);
        }
    for(i=0; i<5; i++) *ui++ = rxbuf[i+1];
    
    } /* end TEI_req */






/*
 *	Function:	TEI_ASS()
 *	Description:    This function assigns a TEI value to a terminal.
 *
 *	Inparameters:   port number
 *			ui frame containing received reference number
 *                      TEI value to be sent
 *                      transmit SAPI index
 *                      transmit TEI index
 *
 *	Outparameters:  none
 *			
 *
 */

TEI_ASS(portnum,ui,tei,t_sapi,t_tei)
int portnum,
    t_sapi,
    t_tei;
byte *ui,
     tei;
    {
    int result;
    setport(portnum);

    set_sapi(t_sapi);     /* management */
    set_tei(t_tei);       /* broadcast  */

    ui[3]  = IDASS;
    ui[4]  = (tei << 1) + 0x01;
    result = trui(ui,5);

    if (result != 0) pr_error(result,"TEI_ASS:  ",STOP);

} /* end TEI_ASS */





/*
 *	Function:	setupLinks()
 *	Description:    This function sets up the layer 2 links
 *                      requested by the structure pointer 
 *                      for the specified port.
 *
 *	Inparameters:   port number
 *                      pointer to link structure
 *
 *	Outparameters:  none
 *			
 *
 */

setupLinks(portnum,links)
int     portnum;
LINK    *links;
    {
    int   i;
    byte  ui[30];
    LINK  *lk;

    setport(portnum);

    /*
     *   SET MANAGEMENT AND BROADCAST
     */

    lk = links;
    do
       {
       if ( ( lk->sapi == MNGMT) AND 
            ( lk->tei  == BC   )     )
            {
            set_rsapi(2,MNGMT);
            set_rntei(2,BC);
            }
       lk++;
       } 
    while( (lk->type != NONE)  AND
           (lk->tei  != NONE)  AND
           (lk->sapi != NONE)     ); 


    /*
     *   SET OTHER LINKS
     */
 
    i = 0;
    lk = links;
    do
        {
        if ( lk->type == AUTOMATIC )
            {
            TEI_req(portnum,ui);
            TEI_ASS(portnum,ui,PHONE_A,2,2);
            }
        if ( lk->tei != BC )
           {
            set_rsapi(0,CONTROL);
            set_rntei(0,PHONE_A);
            set_sapi(0);
            set_tei(0);
            i++;
            }
	lk++;
        }
    while( (lk->type != NONE)  AND
           (lk->tei  != NONE)  AND
           (lk->sapi != NONE)      ); 


} /* end setupLinks */






/*-------------------      end linkCom.c        ----------------*/
