	title 'Character I/O handler '

; CHARIO.ASM
;  Character I/O for the Modular CP/M 3 BIOS
;  This module is organized into routines to do
;  initialization, input, output, and status checks.
;
;  Copyright (c) Visual Technology, 1983
;
;

	maclib Z80	; define Z80 op codes
	maclib ports	; define port addresses


	extrn	?pmsg
	extrn	?bnksl,?wboot
	extrn	@cbnk
	PUBLIC  RSPAR
	public	?cinit,?ci,?co,?cist,?cost
	public	@ctbl
	public	aux$in$ptr,aux$out$ptr, kb$buf$in
	public	dspint, kint, rsint, kmodei

$*MACRO
max$devices	equ 3
ctlz		equ 26
bdos		equ 5
	cseg
	page
;
; ?cinit 
; This module is called for each of the 16 character devices, and
; initializes the devices.  Register C contains the device number.
; The routine initializes the physical character device specified 
; in register C to the baud rate contained in the appropriate entry
; of @ctbl.  This routine need not be supplied if i/o redirection
; is not implemented.  It is referenced only by the DEVICE utility.
;

?cinit:
	lxi	d,xfer$init	; jump table for ?cinit
	mov	b,c		; needs b
	jr   	check$dev	; check for legality

;
; this table parallels @ctbl
;
xfer$init:
	jr	display$init	; init 6502
	jr	aux$init	; init rs-232 port
	jr	kb$init		; init keyboard port
	jr	lpt$init	; init parallel port
kbinit:
	lda	kmode		; Get keyboard bits		(LEL0284)
	out	p$kb$data	; Send out			(LEL0284)

display$init:
lpt$init:
;
; Do nothing (set up in boot rom)                      
;
	ret			;

aux$init:
	;
	; get default baud rate, then set it, by setting the
	; base baud rate in port c of misc. 8255, and baud rate
	; factor in 8251 chip.
	;
	mov	l,b		; get device number
	mvi	h,0		; make into word
	dad h ! dad h ! dad h	; *8 (table entry size)
	lxi	d,@ctbl+7	; character device table
	dad	d		; device baud entry
	mov	l,m		; get baud rate
	mvi	h,0		; make word
	lxi	b,baud$table	; settings to achieve baud rate
	dad	b		; entry in table

	;
	; now have pointer to baud table entry. set base rate.
	;
	di			; disable interrupts
	in	p$misc		; retain old settings
	xra	m		; get new setting
	ani	not misc$baudmask
				; clear base rate bits
	xra	m		; set new ones
	out	p$misc		; reload it

	;
	; now set base rate factor in 8251 chip.
	;
	mvi	a,com$ir	; internal reset
	out	p$aux1$control	; kick it
	mov	a,m		; get new base rate
	xri	mode$default	; default modes
	ani	mode$baudmask	; use baud rate from table
	xri	mode$default	; include defaults
	out	p$aux1$control	; kick it
	mvi	a,com$default	; reset command parameters
	out	p$aux1$control	;  to its default
	push	psw		; Save values			   (LEL0184)
	in	p$aux1$data	; Read in anything		   (LEL0184)
	pop	psw		; Restore command byte		   (LEL0184)
	out	p$aux1$control	; OUT again in case 1st one missed (LEL0184)
	sta	rspar		; Save RS232 parameters
	ei			; enable interrupts

cinit$exit:
	ret
	page
; check$dev
;  Assumes the address of a jump table in de, and a device in b.
;  Preserves b and c.

check$dev:
	mvi	a,max$devices
	cmp	b		; b > max$devices?
	jrc	bad$device	; yes
	mov	l,b		; device to l
	mvi	h,0		; make 16 bits
	dad	h		; double it
	dad	d		; entry in jump table
	pchl

bad$device:
	xra	a
	ret

	page
; ci (b)
;  Character device input. This routine is called with the 
;  character device in register b. It returns an 8 bit
;  character with no parity.

?ci:
	lxi	d,xfer$in
	jr 	check$dev

; must parallel device table
;

xfer$in:
	jr	crtin
	jr	auxin
	jr	kbin
	jr	lptin

crtin:
lptin:
	mvi	a,ctlz		; set to eof character
	ret
auxin:
	lxix	aux$in$ptr	; auxin buffer structure
	call	bufin		; get character
	ei			; enable interrupts
	ret
kbin:

	lxix	kb$buf$in	; Point to keyboard buffer
	call	bufin		; Get character
	ei			; reenable interrupts
	mov	c,a		; Save character in C-reg

;
; Check caps lock
;

	lda	kmode		; Get keyboard mode byte
	ani	010h		; check caps lock
	jrz	kbin$20		; Jump if not on

	mvi	a,'a'-1		; Check if >= 'a'
	cmp	c		;
	jrnc	kbin$20		; Jump if below range
	mvi	a,'z'		; Check if <= 'z'
	cmp	c		;
	jrc	kbin$20		; Jump if above range
	mov	a,c		; Else, get character
	xri	20h		; Shift to upper case
	jr	kbin$exit	; and exit
;
; Check numeric pad
;

kbin$20
	mov	a,c		; Get keycode
	cpi	0bfh		; Numeric pad or less?
	jrnc	kbin$50		; Jump if no 
	
	cpi	83h		; Enter key?
	jrz	kbin$40		; Jump if yes

	cpi	0a0h		; Low end of numeric pad?
	jrc	kbin$50		; Jump if less

kbin$40	lda	kmode		; Test num lock
	ani	20h		;
	jrz	kbin$50		; Jump if not on

	mov	a,c		; Get keycode
	cpi	83h		; Enter key?
	mvi	a,0dh		; (assume yes)
	jrz	kbin$exit	; Jump if true
	res	7,c		; Else change to number
	jr	kbin$55		;

;
; Test other special characters
;

kbin$50	cpi	08ch		; Tab key?
	mvi	a,09h		; Yes => Control-I
	jrz	kbin$exit

kbin$55	mov	a,c		; Get keycode
kbin$exit:
	ret
	page
; bufin (ix)
;  Ix points to buffer data.
;  Returns character in register a, after updating buffer
;  management data structure. Buffer is circular.
;  buf$empty is the offset of the next byte to return.

bufin:
spin:
	di			;
	push	b
	call	?cist 		; is there a character?
	pop	b
	ei			;
	jrz	spin		; no, wait for one

	di			; turn off interrupts
buffer$in:
	ldx	a,buf$empty	; offset of empty in buffer
	ldx	b,buf$length	; length of buffer
	mov	c,a		; save pointer
;
; add 1 mod length
;
	inr	a		; increment buffer pointer
	cmp	b		; did we go out of bounds?
	jrnz	inbounds$i	; if not
	xra	a		; else wrap around

inbounds$i:
	stx	a,buf$empty	; save new emptying pointer
	mvi	a,buf$buffer	; offset of buffer
	add	c		; offset of character
	sta	loc$i		; save in i-stream
	ldx	a,$-$		; load character
loc$i	equ	$-1		; disp. field of ldx
	ret
	page
; cist (b)
;  Character device input status. This routine is called
;  with a device number in register b and returns false
;  if the specified device has no input character ready,
;  and true if the device has an input character to be read.

?cist:

	lxi	d,xfer$st
	jmp	check$dev

; must parallel device table
;

xfer$st:
	jr	crtst
	jr	auxst
	jr	kbst
	jr	lptst

crtst:
lptst:
	xra	a
	ret

auxst:
	lxix	aux$in$ptr	; aux buffer structure
	di			; Disable interrupts
	call	inst		; Anything in buffer?
	ei			; reeable interrupts
	ret			; and return
	page

kbst:
	lxix	kb$buf$in	; keyboard buffer structure
	di			; Disable interrupts
	call	inst		; Check input status
	ei			; reenable interrupts
	rz			; If no character, return

;
; There is a character in the buffer, but in order to know if
; it 'counts', we must get the exact character.  
;

;	lxix	kb$buf$in	; Point to input buffer		(LEL0384)
	di			; disable interrupts
	ldx	b,buf$empty	; get empty pointer
	push	b  		; save it
	call	buffer$in	; Get keycode
	ldx	d,buf$empty	; Get new empty pointer
	pop	b  		; Get empty pointer
	stx	b,buf$empty	; restore it
	ei			; reenable interrupts
	
;
; Now, check for 'non-characters', ie shift and num lock.
;

	lxi	h,kmode		; point to keyboard mode	(LEL0684)
	sui	80h		; Caps lock?
	jrz	kbds$1		; jump if yes
	dcr	a		; Num lock?
	jrz	kbds$2		; jump if yes
	dcr	a		; Num lock up?
	jrz	kbds$4		; jump if yes

	mvi	a,true		; Character is ok
	ret			; 

;
; Handle caps and num lock
;

kbds$1:
;	lda	kmode		; Get keyboard mode		(LEL0684)
	mov	a,m		; Get keyboard mode		(LEL0684)
	xri	010h		; toggle caps lock
	jr	kbds$3		;

kbds$2:
;	lda	kmode		; Get keyboard mode		(LEL0684)
	mov	a,m		; Get keyboard mode		(LEL0684)
	xri	020h		; Toggle numlock bit

kbds$3:	
;	sta	kmode		; Store new settings		(LEL0684)
	mov	m,a		; Store new settings		(LEL0684)
kbds$4:
	di			; disable interrupts
	stx	d,buf$empty	; store empty pointer
	ei			; reenable interrupts
	xra	a		; a=0 for no char
	ret
	page
;inst(ix)
;  Ix is the input buffer structure for a character device.
;  This routine returns true if there is a character waiting,
;  false otherwise.

inst:
	ldx	a,buf$empty	; next byte to empty
	ldx	b,buf$fill	; next byte to fill
	sub	b		; same?
	rz			; yes, it's empty
	mvi	a,true		; return true
	ret
	page
; co (b)
;  Character output. This routine is called with a device number
;  in register b, and the character to be output in c. 
;  It waits until the device (or buffer) is ready to accept 
;  another character, and then sends the character.
;  The character is 8 bits, no parity.

?co:
	lxi	d,xfer$out
	jmp	check$dev

;
; must parallel device table
;

xfer$out:
	jr	disp$out
	jr	aux$out
	jr	kb$out
	jr	lpt$out

kb$out:
	xra	a
	ret
aux$out:
	call	?cost		; device ready for output?
	jrz	aux$out		; loop until ready

	di			; disable interrupts
	call	bufout		; Put char into buffer

	lxi	h,rspar		; Point to RS232 parameters	(LEL0384)
	mov	a,m		; Get them			(LEL0384)
	ori	com$txe		; enables transmit
	mov	m,a		; Store new params		(LEL0384)

	out	p$aux1$control	; aux1 control port
	ei			; reenable interrupts
	ret

lpt$out:
	call	?cost		; busy
	jrz	lpt$out		; yes, spin
	mov	a,c		; get the character
	cma
	out	p$printer
	mvi	a,1		; strobe the printer
	out	p$8255$control	; to 1
	xra	a		; and back
	out	p$8255$control	; to 0
	ret

disp$out:
	call	?cost		; busy?
	jrz	disp$out	; yes, spin
	mov	a,c		; get the character
	out	p$disp$out	; send it out
;
; the following uses the bit set/reset feature
; of the 8255. 
	mvi	a,0000$1110b	; reset port c bit 7
	out	p$disp$control	; to shake hands
	inr	a		; set port c bit 7
	out	p$disp$control	; to complete handshake
	ret
	page
; bufout (ix,c)
;  This routine takes the character to be sent out and
;  stuffs it into the buffer denoted by register ix.

bufout:    
	ldx	a,buf$fill	; fill point in buffer
	ldx	b,buf$length	; length of buffer
	mov	d,a		; save fill point
	inr	a		; next fill point?
	cmp	b		; outside?
	jrnz	inbounds$o	; no.
	xra	a		; yes, reset
inbounds$o:
	stx	a,buf$fill 	; new fill point
inbounds$5:
	mvi	a,buf$buffer	; offset of buffer
	add	d		; old fill point
	sta	loc$o		; next byte to be sent
	stx	c,$-$		; store in buffer
loc$o	equ	$-1
	ret
	page

; cost (b)
;  Character output status. This routine is called with a 
;  device number in register b. It returns with register
;  a set to zero if the device specified cannot accept a
;  character immediately, and returns with ff otherwise.
;  B and C must be preserved.

?cost:
	lxi	d,xfer$ost
	jmp	check$dev

xfer$ost:
	jr	disp$ost
	jr	aux$ost
	jr	kb$ost
	jr	lpt$ost

kb$ost:
	xra	a
	ret
disp$ost:
	in	p$disp$c	; port c
	ani	disp$ready$rcv	; ready to receive?
	rz			; no.
ost$true:

	mvi	a,true		; return ready
	ret

aux$ost:
	lxix	aux$out$ptr	; aux xmit buffer
	di			; critical section
	ldx	d,buf$empty	; next to read
	ldx	a,buf$fill	; next to fill
	ei			; critical section

buf$full$cond:

	; this check is to see if the buffer was full.
	; a full condition is the empty pointer one
	; more than the fill pointer. Empty must always
	; be ahead of fill (modulo  buffer length since
	; circular).

	inr	a
	ldx	e,buf$length	; need to wrap?
	cmp	e		; if equal, wrap
	jrnz	inbound$ost	; not equal
	xra	a

inbound$ost:
	cmp	d		; is fill=empty?
	jrnz	ost$true	; no. not full

ost$false:

	xra	a		; yes. full.
	ret
lpt$ost:
	in	p$misc		; printer status
	ani	misc$lpt$avail	; available?
	jrz	ost$false	; no.
	in	p$misc		; if available
	ani	misc$lpt$nobusy	;   busy?
	jrz	ost$false	; yes
	jr	ost$true	; no

	page

; The entries in the following table are indexed by the
; values for baud rates in modebaud.lib
;
baud$table:
	db	baud$none			; none
	db	baud$none			; 50
	db	baud$none			; 75
	db	baud$none			; 110
	db	baud$none			; 134.5
	db	baud$none			; 150
	db	misc$base01200 or mode$div64	; 300
	db	misc$base02400 or mode$div64	; 600
	db	misc$base01200 or mode$div16	; 1200
	db	baud$none			; 1800
	db	misc$base02400 or mode$div16	; 2400
	db	baud$none			; 3600
	db	misc$base19200 or mode$div64	; 4800
	db	baud$none			; 7200
	db	misc$base09600 or mode$div16	; 9600
	db	misc$base19200 or mode$div16	; 19200


@ctbl	db 'DISPLY'	; device 0, 6502 CRT
	db mb$in$out
	db baud$none
	db 'RS232 '	; device 1, RS-232 port
	db mb$in$out+mb$serial+mb$softbaud
	db baud$1200
	db 'KB    '	; device 2, Keyboard
	db mb$input+mb$serial
;	db baud$1200	;					(LEL0384)
	db baud$600	;					(LEL0384)
	db 'LPT   '	; device 3, Centronics parallel printer
	db mb$output
	db baud$none
	db 0			; table terminator

	page
;
; Keyboard interrupt routine     
;
; Takes incoming character, checks for warm or cold boot sequences.
; If it is either one, it performs them immediately.                
; Otherwise, it stores the keycode in the buffer and performs the keyclick. 
; If there is not enough room in the buffer to store the keycode,
; the key is discarded and the keyboard will beep to indicate an error
; condition.
;
Kint: 
	Push	psw			; Save A
	push	h			; Save H		(LEL0384)
	push	d			; and  d
	push	b			; and b
	pushix				; and ix		(LEL0384)

;
; Get keycode and check for warm or cold boot.       
;
	in	p$kb$data		; get keycode
 	mov	d,a			; save in d

	cpi	08ah			; Is it warm boot?
	jrz	kint$6 			; jump if yes
	cpi	08bh			; Or cold boot?
	jrnz	kint$7 			; jump if not
	page
;
; Key is cold boot.  Jump to boot rom.          
;

kint$2	
	call	reset$disks		; Reset floppy & winch	(LEL0384)

	lxi	h,last$words		; Point to term. seq	(LEL0184)
	mvi	b,Last$length		; Load length		(LEL0184)
kint$3:
	push	b			; Save B		(LEL0184)
	push	h			; Save H		(LEL0184)
	mvi	b,1			; Device=1		(LEL0184)
	mov	c,m			; Get character		(LEL0184)
	call	disp$out		; Send it out		(LEL0184)
	pop	h			; Restore h		(LEL0184)
	pop	b			; Restore B		(LEL0184)
kint$5:
	call	disp$ost	; Wait till done		(LEL1283)
	jrz	kint$5		; (loop if not)			(LEL1283)

	inx	h		; Point to next char		(LEL0184)
	djnz    kint$3		; Finish message		(LEL0184)
	XRA 	A		; A=0
	OUT	P$BOOT		; ENABLE BOOT PROM
	jmp	0
;
; Key is warm boot.
;

kint$6
	call	reset$disks		; Reset floppy & winch	(LEL0384)

	in	p$disk$bits		; Get current settings
	ori	0100$1111b		; Turn off motor & deselect drvs
	out	p$disk$bits		; Send out again

	mvi	a,int$initial
	out	int$port
	ei
	jmp	?wboot 			;  boot


reset$disks:
	mvi	a,fdc$reset		; Reset floppy		(LEL0284)
	out	p$disk$control		; send out		(LEL0284)

	mvi	a,wini$reset		; Reset wini		(LEL0384)
	out	p$winch$control		; send out		(LEL0384)

	mvi	a,1			; Select bank 1		(LEL0384)
	call	?bnksl			; for BDOS call		(LEL0384)
	mvi	c,0dh			; Reset disk system	(LEL0384)
	call	bdos			; perform call		(LEL0384)

	ret				;			(LEL0384)
;
; Key not cold or warm boot.                                       
;
kint$7: 	
;
; If screen is blanked, turn it back on & throw this key away.
;
	lda	blank$flag		; Get blank flag	(LEL0284)
	cpi	0ffh			; Screen blanked?	(LEL0284)
	jrnz	kint$9 			; Jump if not		(LEL0284)
	xra	a			; Clear A		(LEL0284)
	sta	blank$flag		; store as new flag	(LEL0284)
;
; Send NULL to screen to turn it back on
;

	mvi	b,1			; Device #=1		(LEL0284)
	mov	c,a			; C-reg=NUL		(LEL0284)
	call	disp$out		; Send to display	(LEL0284)
	jr	kint$99			; Then exit		(LEL0284)

;
; Screen not blanked.  Put char into buffer.
;
kint$9:
	mov	a,d			; get keycode
kint$11
	push	d			; save keycode
	lxix	kb$buf$in		; point to buffer
	ldx	d,buf$empty		; Get empty pointer
	ldx	a,buf$fill		; and fill pointer
	call	buf$full$cond		; check full condition
	pop	d			; restore keycode
	ora	a			; (checking resultant flag)
	jrnz	kint$30			; jump if true
;
; Keyboard buffer is full.  Beep rather than click and then exit.
;
kint$23
	call	beep    		; beep to indicate error
	jr	kint$99			; go to exit
;
; Keyboard buffer not full.  Put char into buffer.
;
kint$30	mov	c,d			; Get char into C-reg
	push	d			;
	call	bufout			; put into buffer
	pop	psw			; Get character into A
;
; Check for caps lock and num lock
;
	lxi	h,kmodei		; Point to prev values	(LEL0384)

	sui	80h			; Caps lock?		(LEL0384)
	jrz	kint$80			; Jump if yes

	dcr	a  			; Num lock?		(LEL0384)

	jrnz	Kint$99			; Jump if not either   	(LEL0384)
;
; Handle num lock
;

	mov	a,m			; Get previos setting	(LEL0384)
	xri	020h			; Toggle num lock
	jr	kint$90			;			(LEL0384)

;
; Handle caps lock
;

kint$80:
	mov	a,m			; Get previous settings	(LEL0384)
	xri	010h			; Toggle caps lock


;
; Turn on/off keyboard leds.
;

kint$90:
	mov	m,a			; Store new num/caps lk	(LEL0384)
	mov	h,a			; store data

kint$l:
	in	p$kb$control		; Get status
	ani	01h			; Test Tx empty
	jrz	kint$l			; loop if not
	mov	a,h			; Get data back
	out	p$kb$data		; Send to keyboard
;	jr	kint$99			; then exit		(LEL0384)

;
; Get ready to exit
;
kint$99:
	jmp	rs$exit			; (Same as other exit)	(LEL0384)

;	mvi	a,int$initial		; Initialize
;	out	int$port		; Interrupt port
;	popix				; start popping		(LEL0384)
;	pop	b			;
;	pop	d			;
;	pop	h			;			(LEL0384)
;	pop	psw			;
;	ei				; enable interrupts
;	ret				; and return


Last$words:
	db	01bh,'c',00  
last$length	equ	$-last$words
	
	page

;
; Display Interrupt routine     
;
; Takes incoming character from 6502 code.  This info is used for  
; keyboard info only.                                       
;
Dspint:
	Push	psw			; Save A
	push	d			; and d
	push	h			; and h			(LEL0384)

;
; Get character 
;

	in	p$disp$in		; Get character
	mov	d,a 			; save it

	mvi	a,12			; Strobe
	out	p$disp$control		;
	inr	a			; then back
	out	p$disp$control		;

;
; Get character. If it =FF, then it is really a signal that the
; screen has been blanked.
;
	mov	a,d			; Get character		(LEL0284)
	cpi	0FFH			; Is it FF?		(LEL0284)
	jrnz	dspint$44		; Jump if not		(LEL0284)
	sta	blank$flag		; Else, store flag	(LEL0284)
	jr	dspint$77		; and continue		(LEL0284)

;
; Get info.  Mask out the 2 leds. 
;

dspint$44:
	ani	1100$1111b		; minus leds
	mov	d,a			; store in d

	lxi	h,kmode			; Get kmodes		(LEL0384)
	mov	a,m			;			(LEL0384)
	ani	0011$0000b		; Keep leds only
	ora	d			; OR in
	ani	7fh			; (not bell, tho)
	mov	m,a			; Store new values	(LEL0384)


	lxi	h,kmodei		; (both)
	mov	a,m   			;
	ani	0011$0000b		; Keep leds only
	ora	d			;
	mov	m,a   			;

dspint$77:
	out	clear$6502		; Clear 6502 interrupt
	mvi	a,int$initial		; Initialize
	out	int$port		; Interrupt port
	ei				; enable interrupts

dspint$87:
	in	p$kb$control		; Get status
	ani	4			;
	jrz	dspint$87

;
; Now do the actual out to the keyboard
;

	di
	lxi	h,kmodei		; Get byte to send
	mov	a,m   			; 
	out	p$kb$data		; Send out
	ani	7fh			; (don't save beep)
	mov	m,a   			;
	ei

;
; Get ready to exit
;
dspint$99:

	pop	h			; Pop everything	(LEL0384)
	pop	d			;
	pop	psw			;
	ret				; and return


;
; Beep routine
;
beep:
	in	p$kb$control		; Get status
	ani	4			; Test Tx ready
	jrz	beep			; Loop if not
	lda	kmodei			; Get bits
	ori	80h			; Set bell bit
	out	p$kb$data		; Send it out
	ret
	page


;REC/SND (RS-232) INT
;SEE IF RECEIVER OR TRANSMITTER CAUSED INT.
;  IF RECEIVER, ACCEPT CHR INTO RECEIVE FIFO.
;  IF TRANSMITTER, OUT A CHR FROM TRANSMIT FIFO;
;  IF FIFO EMPTY, TURN OFF TRANSMITTER.
RSINT:	PUSH PSW
	push  h
	push  D
	push b
	pushix		;					(LEL0384)

;SEE IF REC OR SND CAUSED INT.

	in p$aux1$control		; read in status
	RAR
	RAR
	jrnC SINT
;
;REC INT: CHR TO FIFO.
RINT:	
	IN p$aux1$data			; read incoming character

	MOV D,A				; save in D
;
; Check for parity error
;
	IN p$aux1$control		; get status
;	bit 3,a				;			(LEL0684)
	ani 0011$1000b			;Any errors?		(LEL0684)
	jrz RINT0			;Jump if ok          

; Parity error.  Reset error condition and replace char with '?'
;
	LDA RSPAR
	ORI 010H
	OUT p$aux1$control
	ANI 0EFH
	OUT p$aux1$control		;RESET ERROR
	MVI D,'?'			;REPLACE CHAR WITH '?'

; Processing continues here.
;
RINT0:			
	push d			; save character
	lxix aux$in$ptr
	ldx d,buf$empty
	ldx a,buf$fill
	call buf$full$cond	; buffer already full?
	pop d			; (restore char)


	JrZ  rs$error		; If yes, error
	mov c,d
	call bufout		; Else, put into buffer
	jr   RS$EXIT


;
;SND INT: CHR FROM FIFO
;
SINT:
	lxix aux$out$ptr	; Point to structure
	call inst		; Anything there?
	jrnz sint2		; Jump if yes
;
; No characters in buffer.  Turn off Xmit enable.
;

sint$1:				;				(LEL0184)
	in p$aux1$control	; Get status			(LEL0184)
	ani 0000$0100b		; Transmitter empty?		(LEL0184)
	jrz sint$1		; Loop till it is		(LEL0184)

	lxi	h,rspar		; Point to RS232 params		(LEL0384)
	mov	a,m		; Get current settings		(LEL0384)
	ANI 0FEH		; Turn off xmit enable
	mov	m,a		; Store new setting		(LEL0384)

	OUT p$aux1$control	;SHUT OFF SND
	jr rs$exit

SINT2:
	call buffer$in		; Get character from buffer
	PUSH	PSW		; Save character		(LEL0184)
sint3:				;				(LEL0184)
	in	p$aux1$control	; Get status			(LEL0184)
;	ani	01h		; Xmit ready?       		(LEL0184)
;	jrz	sint3		; Loop till it is		(LEL0184)

	ani	05h		; Get Xmit ready & empty	(LEL0384)
	cpi	05h		; Both must be set		(LEL0384)
	jrnz	sint3		; To cure double char pblm	(LEL0384)

	POP	PSW		; Restore char			(LEL0184)

	OUT p$aux1$data		; Send it
	jr rs$exit

rs$error:
	mvi	c,'?'		; Load question mark
	ldx	a,buf$fill	; Point to fill location
	ora	a		; Is it 0?
	jrnz	rs$error$5	; Jump if not
	ldx	a,buf$length	; else,wrap
;	jr	rs$error$7	; jump				(LEL0684)

rs$error$5:
	dcr	a		; decrement fill pointer
rs$error$7:
	mov	d,a		;
	call	inbounds$5	; Replace last char with '?'

rs$exit:
	mvi	a,int$initial	; Initialize
	out	int$port	; Interrupt port

	popix			;				(LEL0384)
	pop b
	POP D
	pop h
	POP PSW
	EI
	RET

RSPAR:	DS 1
;
; The keyboard mode byte is kept twice.  One (KMODEI) is used
; by the interrupt routine so that the led can be turned on 
; as soon as the caps or num lock key is hit.  The other
; (kmode) is used to actually effect the change to incoming characters
; as they are being processed.
;

KMODEi:	db 048h			; Initial keyboard 8251 values  (LEL0284)
kmode:	db 048h			;				(LEL0284)
Blank$flag:  db	 0		; 				(LEL0284)



	page
;
; Receive buffers

kb$length	equ	32
kb$buf$in:
	db	kb$length	; length
	db	0		; next character to take
	db	0		; next character to fill
	ds	kb$length	; buffer

aux$length	equ	32
aux$in$ptr:
	db	aux$length	; length
	db	0		; next character to take
	db	0		; next character to fill
	ds	aux$length	; buffer


;
; Transmit buffers

aux$length$o	equ	16
aux$out$ptr:
	db	aux$length$o	; length
	db	0		; next character to take
	db	0		; next character to fill
	ds	aux$length$o	; buffer

bootadd	equ	0

	end
