/*
 * Convergent Technologies
 *
 * FILE: process.c
 *
 * 2.1 C/CTOS
 * Apr  1, 1986  MEC  Created
 *
 * Exercise the CTOS CreateProcess call.
 * Each created process will print one row of asterisks on the screen.
 * Works only in medium model.
 */



#include <stdio.h>



#define CBSTACK    2048			/* Size of new stacks          */
#define FALSE      0
#define NPROC      10			/* Number of created processes */
#define TRUE       0xFF
#define loword(x)  ((short *)(&x))[0]	/* Extract low word of ptr     */
#define hiword(x)  ((short *)(&x))[1]	/* Extract high word of ptr    */
#define uchar      unsigned char
#define uint       unsigned int



typedef struct
{
	int	(*pEntry)();	/* Memory address to begin execution   */
	uint	saData;		/* Segment base address of DS          */
	uint	saExtra;	/* Segment base address of ES          */
	uint	saStack;	/* Segment base address of SS          */
	uint	oStackInit;	/* Initial offset into SS              */
	uchar	priority;	/* Process priority                    */
	uchar	fsys;		/* True if creating a system process   */
	uint	respExch;	/* Default response exchange           */
	uchar	fdebug;		/* True if entering debugger           */
} PCB;



extern	uchar	fDevelopement;	/* Forces CheckErc to call debugger    */



uint	exchExit;		/* Sempahore to count dead children    */
uint	exchRow;		/* Semaphore for locking "row" global  */
uint	exchVideo;		/* Semaphore for locking video         */
PCB	pcb;			/* A process control block             */
char	rgstacks[NPROC][CBSTACK];	/* Stacks -- must be in DGroup */
uint	row;			/* The output row for a child process  */

int	myproc();



/*
 * M A I N
 *
 * Create all the child processes.
 */
main()
{
	auto	uint	i;
	auto	char *	pJunk;
	auto	char *	pStack;
	auto	long	qtime;

	/*
	 * Turn on fDevelopement.  This causes CheckErc to
	 * enter the debugger instead of calling ErrorExit.
	 */
	fDevelopement = TRUE;

	/*
	 * Initialize the random number generator
	 */
	CheckErc(GetDateTime(&qtime));
	srand(loword(qtime));

	/* Grab an exchange for synchronizing exitting */
	CheckErc(AllocExch(&exchExit));

	/* Grab an exchange for synchronizing "row" */
	CheckErc(AllocExch(&exchRow));

	/* Grab an exchange for synchronizing video output */
	CheckErc(AllocExch(&exchVideo));
	unlock();

	/* Clear screen, turn off cursor */
	setbuf(stdout, NULL);
	printf("\377C%c%c\377EF\377VF", 0, 0);

	/* Fire up all the processes */
	for (i = 0; i < NPROC; i++)
	{
		/* Set the row number for the created process. */
		row = i + 10;

		/*
		 * Set up the Process Control Block for the child.
		 *
		 * pEntry is the address of the procedure that the
		 * child process will enter.
		 *
		 * saData must equal the selector of DGroup.
		 * saExtra can be anything.
		 * saStack must equal the selector of DGroup in order
		 * for CTOS Object Module Procedures (such as byte
		 * streams) to work.  In particular, this means the
		 * stack cannot come from AllocMemorySL() or malloc()
		 * (if you want to use CTOS Object Module Procedures).
		 *
		 * oStackInit is the top of the stack (not the bottom!)
		 * priority is set to a reasonable default.
		 * fsys must be FALSE in user programs.
		 * respExch must be a uniquely allocated exchange.
		 * fdebug is generally FALSE.
		 */
		pStack         = &rgstacks[i][CBSTACK];
		pcb.pEntry     = myproc;
		pcb.saData     = hiword(pStack);
		pcb.saExtra    = hiword(pStack);
		pcb.saStack    = hiword(pStack);
		pcb.oStackInit = loword(pStack);
		pcb.priority   = 0x80;
		pcb.fsys       = FALSE;
		CheckErc(AllocExch(&pcb.respExch));
		pcb.fdebug     = FALSE;
		CheckErc(CreateProcess(&pcb));

		/* Wait for the process to pick up its row number */
		CheckErc(Wait(exchRow, &pJunk));
	}

	/*
	 * Wait for all the children to "terminate".
	 */
	for (i = 0; i < NPROC; i++)
	{
		CheckErc(Wait(exchExit, &pJunk));
	}
}



/*
 * M Y P R O C
 *
 * Each child process enters myproc.
 * Print a row of stars across the screen, one at a time.
 */
myproc()
{
	auto	uint	i;
	auto	uint	ndelay;
	auto	uint	myExch;
	auto	uint	myRow;
	auto	char *	pJunk;

	/*
	 * The row number is kept in a global variable.
	 * Pick it up and wakeup the parent process.
	 */
	myRow = row;
	CheckErc(Send(exchRow, &i));

	/*
	 * Write a row of '*' in the middle of the screen.
	 * After each character, pause a random amount of time.
	 * rand() isn't reentrant, so it must be called inside lock.
	 */
	for (i = 20; i < 60; i++)
	{
		lock();
		printf("%c%c%c%c%c", 0xFF, 'C', i, myRow, '*');
		ndelay = rand();
		unlock();
		CheckErc(delay(ndelay % 4));
	}

	/*
	 * Let the parent process know we're done
	 */
	CheckErc(Send(exchExit, &i));

	/*
	 * Fall asleep
	 */
	CheckErc(QueryDefaultRespExch(&myExch));
	CheckErc(Wait(myExch, &pJunk));
	CheckErc(8);
}



/*
 * L O C K
 *
 * Obtain exclusive access to the video.
 */
lock()
{
	auto	char *	pJunk;
	CheckErc(Wait(exchVideo, &pJunk));
}



/*
 * U N L O C K
 *
 * Relinquish access to the video.
 */
unlock()
{
	auto	char *	pJunk;
	CheckErc(Send(exchVideo, &pJunk));
}
