/*
	Copyright 1987-1990 XVT Software Inc. All rights reserved.
	May be used freely by licensed and registered users of XVT.
	May be distributed in source form only when embedded in an
	XVT user's application.
*/

#include "xvt.h"					/* standard XVT header */
#if (XVTCC == MWCCC) && (XVTOS == CTOOS)
pragma Calling_convention(CTOS_CALLING_CONVENTIONS);
#endif
#include "xvtmenu.h"				/* standard XVT menu tags */

/*
	This is an example program that shows how to use application data to
	dispatch events directly to a window, and how windows can be treated
	as objects.

	This example creates windows of two classes. Class A windows display
	the heading "Class A" and the text unique to the window; class B
	windows display the heading "Class B" and the text.

	The class of a window determines its general characteristics; in a
	fancy desktop publishing application the classes might be "textual
	document" and "drawing." In general each textual-document window would
	contain a different document, and each drawing window would contain a
	different drawing. So, in addition to its common class characteristics,
	a window has some instance data, too.
                                                                                
	This terminology is meant to parallel that used in object-oriented
	programming systems such as C++. The class of a window determines
	the operations it can perform, whereas each window (object) is an
	instance of that class and has its own data.

	In this example, each window is associated with an object of type
	DISPATCH with two members: class_fcn is a pointer to the function that
	handles events for its class, and instance_data is a pointer to the
	window's data.

	One interesting, and valuable, feature of this example is its total
	absence of global variables (with the exception of appl_setup,
	of course). The WINDOW returned by new_window is not only not assigned
	to a global variable--it is actually thrown away!
*/

/*
	The following two types are for the class_fcn member of the DISPATCH
	structure and for DISPATCH itself.
*/
#ifdef PROTO
typedef void (XVTENTRY *EVENT_HANDLER) BTCENTRY(WINDOW win, EVENT *ep, char *data);
#else
typedef void (XVTENTRY *EVENT_HANDLER) BTCENTRY();
#endif
typedef struct {
	EVENT_HANDLER class_fcn;		/* fcn to handle events */
	char *instance_data;			/* data specific to window */
} DISPATCH;

#ifdef PROTO
STATICFCN void XVTENTRY class_A_handler BTCENTRY(WINDOW win, EVENT *ep, char *data);
STATICFCN void XVTENTRY class_B_handler BTCENTRY(WINDOW win, EVENT *ep, char *data);
STATICFCN void build_window(int left, int top, int right, int bottom, char class,
  char *data);
#else
STATICFCN void XVTENTRY class_A_handler BTCENTRY();
STATICFCN void XVTENTRY class_B_handler BTCENTRY();
STATICFCN void build_window();
#endif

/*
	Function to receive all events for windows of class A. The response
	to events determines its class properties.
*/
#ifdef FPROTO
STATICFCN void XVTENTRY class_A_handler BTCENTRY(WINDOW win, EVENT *ep, char *data)
#else
STATICFCN void XVTENTRY class_A_handler BTCENTRY(win, ep, data)
WINDOW win;							/* window receiving event */
EVENT *ep;							/* event info */
char *data;							/* data unique to window */
#endif
{
	RCT rct;

	switch (ep->type) {
	case E_UPDATE:
		get_client_rect(win, &rct);
		set_cpen(&hollow_cpen);
		set_cbrush(&white_cbrush);
		draw_rect(&rct);
		draw_text(10, 50, "I am a window of class A", -1);
		draw_text(10, 75, "My data is:", -1);
		draw_text(10, 100, data, -1);
		break;
	case E_MOUSE_DBL:
		note("Class A window with data \"%s\" was double-clicked.", data);
		break;
	case E_CLOSE:
		close_window(win);
		break;
	case E_KILL_WINDOW:
		free((char *)get_app_data(win));
	}
}

/*
	Function to receive all events for windows of class B. This function
	is almost identical to class_A_handler in this example. In a
	real application the two functions would be as different as the classes
	are different.
*/
#ifdef FPROTO
STATICFCN void XVTENTRY class_B_handler BTCENTRY(WINDOW win, EVENT *ep, char *data)
#else
STATICFCN void XVTENTRY class_B_handler BTCENTRY(win, ep, data)
WINDOW win;							/* window receiving event */
EVENT *ep;							/* event info */
char *data;							/* data unique to window */
#endif
{
	RCT rct;

	switch (ep->type) {
	case E_UPDATE:
		get_client_rect(win, &rct);
		set_cpen(&hollow_cpen);
		set_cbrush(&white_cbrush);
		draw_rect(&rct);
		draw_text(10, 50, "I am a window of class B", -1);
		draw_text(10, 75, "My data is:", -1);
		draw_text(10, 100, data, -1);
		break;
	case E_MOUSE_DBL:
		note("Class B window with data \"%s\" was double-clicked.", data);
		break;
	case E_CLOSE:
		close_window(win);
		break;
	case E_KILL_WINDOW:
		free((char *)get_app_data(win));
	}
}

APPL_SETUP appl_setup = {
	0,								/* menu bar resource ID (use default) */
	0,								/* about box resource ID (use default) */
	"dispatch",						/* application's name */
	W_NONE,							/* type of initial window */
	FALSE,							/* size box on initial window? */
	FALSE,							/* vert. scroll bar on initial window? */
	FALSE,							/* horz. scroll bar on initial window? */
	FALSE,							/* close box on initial window? */
	FALSE,							/* want std. font menu? (includes sizes) */
	FALSE							/* want std. style menu? */
};

/*
	Function to build a window with a given client-area rectangle, class, and
	data. Note that since each window is associated with its event handler via
	its application data, the WINDOW tag returned by new_window doesn't have to
	be kept around.
*/
#ifdef FPROTO
STATICFCN void build_window(int left, int top, int right, int bottom, char class,
  char *data)
#else
STATICFCN void build_window(left, top, right, bottom, class, data)
int left, top, right, bottom;		/* rectangle coordinates */
char class;							/* class (A or B) */
char *data;							/* window's data */
#endif
{
	WINDOW win;
	RCT rct;
	DISPATCH *dp;
	static int count = 0;
	char title[25];

	if ((dp = (DISPATCH *)malloc(sizeof(DISPATCH))) == NULL) {
		error("Insufficient memory.");
		return;
	}
	set_rect(&rct, left, top, right, bottom);
	sprintf(title, "Window %d", ++count);
	if ((win = new_window(&rct, title, W_DOC, TRUE, FALSE, FALSE, TRUE)) ==
	  NULL_WIN)
		error("Can't create window %d", count);
	else {
		dp->class_fcn = class == 'A' ? class_A_handler : class_B_handler;
		dp->instance_data = data;
		set_app_data(win, PTR_LONG(dp));
	}
}

/*
	Function called by XVT to initialize the application. It just creates a
	couple of windows of each class.
*/
BOOLEAN XVTENTRY appl_init BTCENTRY(void)
{
	build_window(5, 50, 205, 225, 'A', "elephant");
	build_window(25, 70, 225, 245, 'A', "zebra");
	build_window(260, 50, 460, 225, 'B', "tiger");
	build_window(280, 70, 480, 245, 'B', "giraffe");
	return(TRUE);
}

/*
	Main event function. Note how window-oriented events are dispatched to
	the appropriate class function.
*/
#ifdef PROTO
void XVTENTRY main_event BTCENTRY(WINDOW win, EVENT_PTR ep)
#else
void XVTENTRY main_event BTCENTRY(win, ep)
WINDOW win;
EVENT_PTR ep;
#endif
{
	DISPATCH *dp;

	if (win != NULL_WIN && (dp = (DISPATCH *)get_app_data(win)) != NULL)
		(*dp->class_fcn)(win, ep, dp->instance_data);
	else
		switch (ep->type) {
		case E_COMMAND:
			if (ep->v.cmd.tag == M_FILE_QUIT)
				terminate();
			break;
		case E_QUIT:
			if (ep->v.query)
				quit_OK();
			else
				terminate();
		}
}

/* Empty function; no cleanup needed. */
void XVTENTRY appl_cleanup BTCENTRY(void)
{
}

/* End. */
