/*
	Copyright 1987-1990 XVT Software Inc. All rights reserved.
	May be used freely by licensed and registered users of XVT.
	May be distributed in source form only when embedded in an
	XVT user's application.

    Example XVT application, showing how to put data onto and get data off
    of the clipboard.
*/

#include "xvt.h"                    /* standard XVT header */
#include "xvtmenu.h"                /* standard menu tags */
#include "clip.h"                   /* application's menu tags and dialog IDs */

#ifdef PROTO
STATICFCN BOOLEAN XVTENTRY cb_fmt(int, CONTROL_INFO *);
STATICFCN void free_data(void);
STATICFCN BOOLEAN fmt_dialog(void);
STATICFCN void clip_get(BOOLEAN);
STATICFCN void put_picture(void);
STATICFCN void put_text(void);
STATICFCN void put_appl(char *);
STATICFCN void clip_put(void);
STATICFCN void show_text(void);
STATICFCN void update(void);
STATICFCN void paste(void);
STATICFCN void do_menu(MENU_TAG, BOOLEAN, BOOLEAN);
STATICFCN void update_menus(void);

#else
STATICFCN BOOLEAN XVTENTRY cb_fmt();
STATICFCN void free_data();
STATICFCN BOOLEAN fmt_dialog();
STATICFCN void clip_get();
STATICFCN void put_picture();
STATICFCN void put_text();
STATICFCN void put_appl();
STATICFCN void clip_put();
STATICFCN void show_text();
STATICFCN void update();
STATICFCN void paste();
STATICFCN void do_menu();
STATICFCN void update_menus();
#endif

/* Required application setup structure. */
APPL_SETUP appl_setup = {
    0,                              /* menu bar resource ID (use default) */
    0,                              /* about box resource ID (use default) */
    "clip",                         /* application's name */
    W_DOC,                          /* type of initial window */
    TRUE,                           /* size box on initial window? */
    FALSE,                          /* vert. scroll bar on initial window? */
    FALSE,                          /* horz. scroll bar on initial window? */
    FALSE,                          /* close box on initial window? */
    FALSE,                          /* want std. font menu? (includes sizes) */
    FALSE                           /* want std. style menu? */
};

#define SZ_APPL_NAME	4			/* max size of CB_APPL format name */
#define SZ_TEXT			20000		/* max size of gotten text data */
#define APPL_FORMAT		"ABCD"		/* our personal clipboard format */

/*
	Application data format is an int representing the string length,
	followed by an array of characters of that length.
	The application cannot rely on the sizep argument to cb_get() as
	the exact size of the data.  The sizep argument IS guarenteed to
	be at least as large as the amount of sotrage needed to hold the data.
*/

/*
	Following structure contains the user's current format options,
	changed only when the OK button in the format dialog is clicked.
*/
static struct s_fmt {				/* effective format options */
	CB_FORMAT fmt;					/* format */
	char name[SZ_APPL_NAME + 1];	/* name, if CB_APPL */
} format = {
	CB_TEXT,
	""
};
static int fmt_dismiss;				/* button that dismissed format dialog */

/* Following structure contains the current data gotten from the clipboard. */
static struct s_data {
	BOOLEAN valid;					/* valid data? */
	CB_FORMAT fmt;					/* format */
	long size;						/* size of data */
	char *ptr;						/* pointer to text or appl data */
	PICTURE pict;					/* picture, if CB_PICT */
	RCT frame;						/* framing rectangle, if CB_PICT */
} data;

static CB_FORMAT paste_fmt;			/* format for Paste on Edit menu */

/*
	Callback function for format dialog box.  For additional comments on the
	logic of callback functions, see the example application XVT-Example
	(file examp.c).
*/
#ifdef  FPROTO
STATICFCN BOOLEAN XVTENTRY cb_fmt(int control_id, CONTROL_INFO *cip)
#else
static BOOLEAN cb_fmt XVTENTRY(control_id, cip)
int control_id;                     /* ID of control that got activated */
CONTROL_INFO *cip;
#endif
{
	static struct s_fmt tmp_fmt;
	int button;
	char name[25];

	NOREF(cip);
	fmt_dismiss = control_id;
	switch (control_id) {
	case DLG_INIT:
		tmp_fmt = format;
		set_item_text(FMT_NAME, tmp_fmt.name);
		select_item_text(FMT_NAME, 0, 32767);
		break;
	case DLG_OK:
		if (tmp_fmt.fmt == CB_APPL) {
			get_item_text(FMT_NAME, name);
			if (name[0] == '\0' || strlen(name) > SZ_APPL_NAME) {
				note("You must specify a format name of 1 to %d characters.",
				  SZ_APPL_NAME);
				select_item_text(FMT_NAME, 0, 32767);
				return(TRUE);
			}
			strcpy(tmp_fmt.name, name);
		}
		format = tmp_fmt;
		return(FALSE);
	case DLG_CANCEL:
		return(FALSE);
	case FMT_TEXT:
		tmp_fmt.fmt = CB_TEXT;
		break;
	case FMT_PICT:
		tmp_fmt.fmt = CB_PICT;
		break;
	case FMT_APPL:
		select_item_text(FMT_NAME, 0, 32767);
		/* fall through */
	case FMT_NAME: /* typing something selects the Application radio button */
		tmp_fmt.fmt = CB_APPL;
		break;
	}
	switch (tmp_fmt.fmt) {
	case CB_TEXT:
		button = FMT_TEXT;
		break;
	case CB_PICT:
		button = FMT_PICT;
		break;
	case CB_APPL:
		button = FMT_APPL;
		break;
	default:
		fatal("Invalid format [cb_fmt].");
	}
	check_radio_button(button, FMT_TEXT, FMT_APPL);
	return(TRUE);
}

/* Function to free saved data, if any. */
#ifdef FPROTO
STATICFCN void free_data(void)
#else
static void free_data()
#endif
{
	if (data.pict != NULL_WIN) {
		picture_free(data.pict);
		data.pict = NULL_WIN;
	}
	if (data.ptr != NULL) {
		free(data.ptr);
		data.ptr = NULL;
	}
	data.valid = FALSE;
}

/* Function to put up the format dialog box. */
#ifdef FPROTO
STATICFCN BOOLEAN fmt_dialog(void)
#else
static BOOLEAN fmt_dialog()
#endif
{
	if (!new_dialog(DLG_MODAL, FMT_DLG, cb_fmt, 0L)) {
		error("Can't put up dialog.");
		return(FALSE);
	}
	return(fmt_dismiss == DLG_OK);
}

/* Function to get data from the clipboard. */
#ifdef FPROTO
STATICFCN void clip_get(BOOLEAN user_initiated)
#else
static void clip_get(user_initiated)
BOOLEAN user_initiated;
#endif
{
	char far *p;

	if (!cb_open(FALSE)) {
		error("Error opening clipboard.");
		return;
	}
	free_data();
	invalidate_rect(std_win, NULL);
	data.fmt = format.fmt;
	if ((p = cb_get(data.fmt, format.name, &data.size)) == NULL) {
		if (user_initiated)
			note("No data in chosen format.");
	}
	else
		switch (data.fmt) {
		case CB_TEXT:
		case CB_APPL:
			if (data.size > SZ_TEXT) {
				error("More than %d bytes of data on clipboard.", SZ_TEXT);
				break;
			}
			if ((data.ptr = malloc((unsigned)data.size)) == NULL) {
				error("Can't allocate memory.");
				break;
			}
			gmemcpy(data.ptr, p, data.size);
			data.valid = TRUE;
			break;
		case CB_PICT:
			if ((data.pict = picture_make(p, data.size, &data.frame)) == NULL_WIN) {
				error("Error making picture.");
				break;
			}
			data.valid = TRUE;
		}
	if (!cb_close())
		error("Error closing clipboard.");
}

/* Function to draw a picture and put it onto the clipboard. */
#ifdef FPROTO
STATICFCN void put_picture(void)
#else
static void put_picture()
#endif
{
	PICTURE pict;
	RCT frame;
	CBRUSH cbrsh;
	
	cbrsh.pat = PAT_DIAGCROSS;
	cbrsh.color = COLOR_BLACK;

	set_rect(&frame, 20, 20, 100, 100);
	if (!picture_open(&frame))
		error("Error opening picture.");
	set_cbrush(&cbrsh);

	set_cpen(&black_cpen);
	draw_oval(&frame);
	if ((pict = picture_close()) == NULL_WIN)
		error("Error closing picture.");
	if (!cb_put(CB_PICT, NULL, 0L, pict)) /* will free PICTURE */
		error("Error putting picture onto clipboard.");
}

/*
	Function to put some text onto the clipboard. Although the text is static
	here, the algorithm to put it onto the clipboard is general, and can be
	used for any collection of text lines.
*/
#ifdef FPROTO
STATICFCN void put_text(void)
#else
static void put_text()
#endif
{
	char far *p;
	static char *text[] = {
		"The quick brown fox",
		"jumped over the",
		"lazy dogs.",
		NULL
	};
	int i, j, eol_len;
	long size;

	eol_len = strlen(EOL_SEQ);
	size = 0;
	for (i = 0; text[i] != NULL; i++)
		size += strlen(text[i]) + eol_len;
	if ((p = cb_malloc(size)) == NULL) {
		error("Can't allocate clipboard memory.");
		return;
	}
	for (i = 0; text[i] != NULL; i++) {
		for (j = 0; text[i][j] != '\0'; j++)
			*p++ = text[i][j];
		for (j = 0; EOL_SEQ[j] != '\0'; j++)
			*p++ = EOL_SEQ[j];
	}
	if (!cb_put(CB_TEXT, NULL, size, (PICTURE)NULL))
		error("Error putting text onto clipboard.");
	cb_free();
}

/* Hokey function to put some application data onto the clipboard. */
#ifdef FPROTO
STATICFCN void put_appl(char *name)
#else
static void put_appl(name)
char *name;
#endif
{
	char far *p;
	int i;
	long size;

	size = 25 + sizeof(int);
	if ((p = cb_malloc(size)) == NULL) {
		error("Can't allocate clipboard memory.");
		return;
	}
	*(int far *)p = 25;
	p += sizeof(int);
	for (i = 0; i < 25; i++)
		*p++ = (char)('A' + i);
	if (!cb_put(CB_APPL, name, size, (PICTURE)NULL))
		error("Error putting data onto clipboard.");
	cb_free();
}

/* Function to put data onto the clipboard. */
#ifdef FPROTO
STATICFCN void clip_put(void)
#else
static void clip_put()
#endif
{
	if (!cb_open(TRUE)) {
		error("Error opening clipboard.");
		return;
	}
	switch (format.fmt) {
	case CB_TEXT:
		put_text();
		break;
	case CB_PICT:
		put_picture();
		break;
	case CB_APPL:
		put_appl(format.name);
		break;
	default:
		error("Invalid format [clip_put].");
	}
	if (!cb_close())
		error("Error closing clipboard.");
}

/* Function to show clipboard text on the standard window, during update.*/
#ifdef FPROTO
STATICFCN void show_text(void)
#else
static void show_text()
#endif
{
	int start, i, ascent, leading, y;

	set_font(&normal_font, FALSE);
	get_font_metrics((INTPTR)&leading, (INTPTR)&ascent, NULL);
	for (i = start = y = 0; ; i++)
        if (data.ptr[i] == '\r' || data.ptr[i] == '\n' || i >= (int)data.size) {
			y += ascent + leading;
			draw_text(10, y, &data.ptr[start], i - start);
			if (i >= (int)data.size)
				break;
			if (data.ptr[i + 1] == '\n')
				i++;
			start = i + 1;
		}
}

/* Function to update the standard window. */
#ifdef FPROTO
STATICFCN void update(void)
#else
static void update()
#endif
{
	RCT rct;
	int size;
	char buf[100];
	
	get_client_rect(std_win, &rct);
	set_font(&normal_font, FALSE);
	set_cpen(&white_cpen);
	set_cbrush(&white_cbrush);
	draw_rect(&rct);
	if (data.valid)
		switch (data.fmt) {
		case CB_PICT:
			picture_draw(data.pict, &data.frame);
			break;
		case CB_TEXT:
			show_text();
			break;
		case CB_APPL:
			sprintf(buf, "Application data in %s format:", format.name);
			draw_text(10, 20, buf, -1);
			set_font(&small_font, FALSE);
			size = *(int *)data.ptr;
			if (size > 100)
				size = 100;
			draw_text(10, 40, data.ptr + sizeof(int), size);
			break;
		default:
			error("Invalid clipboard format [update].");
		}
	else
		draw_text(10, 20, "No data was gotten from the clipboard.", -1);
}

/* Function to perform the Paste command. */
#ifdef FPROTO
STATICFCN void paste(void)
#else
static void paste()
#endif
{
	format.fmt = paste_fmt;
	strcpy(format.name, APPL_FORMAT);
	clip_get(FALSE);
}

/* Function to handle all menu commands. */
#ifdef FPROTO
STATICFCN void do_menu(MENU_TAG cmd, BOOLEAN shift, BOOLEAN control)
#else
static void do_menu(cmd, shift, control)
MENU_TAG cmd;						/* menu tag */
BOOLEAN shift;						/* was shift key down? */
BOOLEAN control;					/* was control key down? */
#endif
{
	NOREF(shift);
	NOREF(control);
	switch (cmd) {
	case M_FILE_CLOSE:
	case M_FILE_QUIT:
		terminate();
		break;
	case M_EDIT_PASTE:
		paste();
		break;
	case M_EDIT_CLIPBOARD:
	case M_CLIP_GET:
		if (fmt_dialog())
			clip_get(TRUE);
		break;
	case M_CLIP_PUT:
		if (fmt_dialog())
			clip_put();
		break;
	default:
		return;
	}
}

/*
	Function to update menus each time this application is activated.  Its
	main role is to enable or disable Paste and Show Clipboard on the Edit menu.
	Here we establish our precedence: our own format, then a picture, then text.
*/
#ifdef FPROTO
STATICFCN void update_menus(void)
#else
static void update_menus()
#endif
{
	BOOLEAN paste_enable = TRUE;

	if (cb_format_avail(CB_APPL, APPL_FORMAT))
		paste_fmt = CB_APPL;
	else if (cb_format_avail(CB_PICT, NULL))
		paste_fmt = CB_PICT;
	else if (cb_format_avail(CB_TEXT, NULL))
		paste_fmt = CB_TEXT;
	else
		paste_enable = FALSE;
	menu_enable(M_EDIT_PASTE, paste_enable);
	menu_enable(M_EDIT_CLIPBOARD, paste_enable);
}

/* Application initialization. */
BOOLEAN XVTENTRY appl_init BTCENTRY (void)
{
#if XVTWS == WSWM
	fatal("The Clip example is not supported for XVT/CH\n");
#endif	
	update_menus();
	paste();
	return(TRUE);
}

/* Main application entry point. */
#ifdef PROTO
void XVTENTRY main_event BTCENTRY(WINDOW win, EVENT_PTR ep)
#else
void XVTENTRY main_event BTCENTRY(win, ep)
WINDOW win;							/* window */
EVENT_PTR ep;							/* event information */
#endif
{
	NOREF(win);
	switch (ep->type) {
	case E_UPDATE:
		update();
		break;
	case E_ACTIVATE:
		if (ep->v.active)
			update_menus();
		break;
	case E_COMMAND:
		do_menu(ep->v.cmd.tag, ep->v.cmd.shift, ep->v.cmd.control);
		break;
	case E_CLOSE:
		terminate();
		break;
	case E_QUIT:
		if (ep->v.query)
			quit_OK();
		else
			terminate();
		break;
	}
}

/* Application cleanup.  Nothing to do. */
void XVTENTRY appl_cleanup BTCENTRY(void)
{
}
