/*****************************  MODULE HEADER  *******************************/
/*                                                                           */
/*                                                                           */
/*  MACHINE:                LANGUAGE:  Metaware C            OS: CTOS        */
/*                                                                           */
/*  file_headers.c                                                           */
/*                                                                           */
/*                                                                           */
/*  Functions to verify CTOS disk volume File Headers.                       */
/*                                                                           */
/*  HISTORY:                                                                 */
/*  --------                                                                 */
/*                                                                           */
/*  MM/DD/YY  VVVV/MM  PROGRAMMER    /  DESCRIPTION                          */
/*                                                                           */
/*  04/29/92  122H.09  D. Gilson     /  PLE 15469544 - Total # of files is   */
/*                                      not correct when multi vols backups  */
/*  03/20/92  122G.08  D. Gilson     /  PLE 15419300 - EofLfa != Sum of runs */
/*                                      files are missing.                   */
/*  07/18/91  130A.07  P. Johansson  /  Verify that all free FHB's are on    */
/*                                      the free list.                       */
/*  05/29/91  121J.06  P. Johansson  /  Count a bypassed file only if the    */
/*                                      file header is an initial FHB!       */
/*  05/10/91  121J.05  P. Johansson  /  Count the bypassed files.            */
/*  05/01/91  121J.04  P. Johansson  /  "Processing file header n of N ... " */
/*  03/28/91  121H.03  P. Johansson  /  Support multiple volume backup to    */
/*                                      [XXX]+ datasets on tape.             */
/*  01/04/90  121F.02  P. Johansson  /  Check both creation and modification */
/*                                      dates if incremental backup.         */
/*  12/27/90  121F.01  P. Johansson  /  Fixed display of volume structures   */
/*  11/30/90  121E.00  P. Johansson  /  Created.                             */
/*                                                                           */
/*                    PROPRIETARY  PROGRAM  MATERIAL                         */
/*                                                                           */
/*  THIS MATERIAL IS PROPRIETARY TO UNISYS CORPORATION AND IS NOT TO         */
/*  BE REPRODUCED, USED OR DISCLOSED EXCEPT IN ACCORDANCE WITH PROGRAM       */
/*  LICENSE OR UPON WRITTEN AUTHORIZATION OF THE PATENT DIVISION OF          */
/*  UNISYS CORPORATION, DETROIT, MICHIGAN 48232, USA.                        */
/*                                                                           */
/*  COPYRIGHT (C) 1990 UNISYS CORPORATION. ALL RIGHTS RESERVED               */
/*                                                                           */
/*****************************************************************************/
/*                                                                           */
/*  UNISYS BELIEVES THAT THE SOFTWARE FURNISHED HEREWITH IS ACCURATE         */
/*  AND RELIABLE, AND MUCH CARE HAS BEEN TAKEN IN ITS PREPARATION. HOWEVER,  */
/*  NO RESPONSIBILITY, FINANCIAL OR OTHERWISE, CAN BE ACCEPTED FOR ANY       */
/*  CONSEQUENCES ARISING OUT OF THE USE OF THIS MATERIAL, INCLUDING LOSS     */
/*  OF PROFIT, INDIRECT, SPECIAL, OR CONSEQUENTIAL DAMAGES, THERE ARE NO     */
/*  WARRANTIES WHICH EXTEND BEYOND THE PROGRAM SPECIFICATION.                */
/*                                                                           */
/*  THE CUSTOMER SHOULD EXERCISE CARE TO ASSURE THAT USE OF THE SOFTWARE     */
/*  WILL BE IN FULL COMPLIANCE WITH LAWS, RULES AND REGULATIONS OF THE       */
/*  JURISDICTIONS WITH RESPECT TO WHICH IT IS USED.                          */
/*                                                                           */
/**************************  END OF MODULE HEADER  ***************************/

#ifdef debug
#define private
#else
#define private static
#endif

/* Standard C library macros and functions invoked by this module */

pragma Off(List);
#include <intel80X86.h>
#include <string.h>
pragma Pop(List);

/* There are no procedures in the Sequential Access service that can cope with
   a variable number of arguments, so this pragma makes everything much more
   efficient.  However, it has to be established AFTER any standard C library
   functions are defined because it reverses the normal C convention. */

pragma Calling_convention(_CALLEE_POPS_STACK);

/* External CTOS and CTOS Toolkit functions invoked by this module */

#define BuildFileSpec
#define NlsULCmpB
#define Send
#define Wait

pragma Off(List);
#include <ctoslib.h>
pragma Pop(List);

#if defined(debug) && defined(breakpoint)
#undef breakpoint
extern void breakpoint(unsigned debug_value_for_AX);
#endif

/* Type definitions used by this module */

#define last(array) (sizeof(array) / sizeof(*array) - 1)

#define FhbType
#define sdType
#define vhb_type

pragma Off(List);
#include <ctosTypes.h>
#include <ext_ctos_types.h>
#include "archive.h"
#include "archive_msgs.h"
pragma Pop(List);

/* Other external functions in this application invoked by this module */

extern iob_type *allocate_iob(unsigned buffer_size);
extern void create_archive_EOD(Boolean final_close);
extern void create_archive_sentinel(unsigned long page);
extern void hex_dump(void *buffer, unsigned length, unsigned blackouts,
                     unsigned blackout_floor[], unsigned blackout_ceiling[]);
extern log_msg(unsigned nls_msg_index, unsigned signature, sdType nls_parms[],
               unsigned nls_parms_len);
extern void match_directory_entry_to_fhb(void);
extern unsigned read_pages(void *buffer, unsigned long vda,
                           unsigned page_offset, unsigned pages);
extern void release_disk_run(unsigned long vda, unsigned long length);
extern void update_path(void);
extern vid_only_msg(unsigned nls_msg_index, unsigned signature,
                    sdType nls_parms[], unsigned nls_parms_len);

/* Error return codes used by this module */

#define FsErc

pragma Off(List);
#include <erc.h>
pragma Pop(List);

/* External variables imported by this module */

extern archive_type archive;
extern filesys_type filesys;
extern path_type path;
extern summary_type summary;
extern target_type target;
extern vhb_type vhb;
extern vlpb_type vlpb;

/* Global variables exported by this manuscript */

FhbType fhb;

/* Static variables global within this manuscript */

private unsigned unused_fhbs = 0;

/* Function prototypes defined before the functions themselves are declared */

void archive_disk_run(unsigned long vda, unsigned long run_length);
void display_fhb(FhbType *fhb, Boolean hex);
unsigned long find_file_length (unsigned page);
unsigned read_fhb(unsigned fhb_num);
unsigned valid_fhb(unsigned page, FhbType *fhb, Boolean alternate);

pragma Page(1);
/*-----------------------------------------------------------------------------
 This procedure is the core of the Volume Archive processing.  Starting with
 file header 1 (the first, file header zero, is always reserved, never used)
 all of the file headers are processed in order. */

void backup_and_or_verify_fhbs(Boolean final_close) {

   Boolean bypass_this_file;
   unsigned erc, free_fhbs = 0, i, j, page, sequence;
   sdType nls_parms[] = {(void *) &summary.count[CURRENT_FILE_HEADER],
                                  sizeof(summary.count[CURRENT_FILE_HEADER]),
                         (void *) &summary.count[TOTAL_FILE_HEADERS],
                                  sizeof(summary.count[TOTAL_FILE_HEADERS]),
                         (void *) &target.filespec[1], 0};

   unused_fhbs = 0;					/* reset for next volume m09 */
   if (!vlpb.suppress_backup)
      log_msg(NLS_NEWLINE, 1, NULL, 0);
   for (i = 1; i < vhb.cPagesFileHeaders; i++) {
      if (     vhb.altFileHeaderPageOffset == 0
            || (i / vhb.altFileHeaderPageOffset & 1) == 0) {
         summary.count[CURRENT_FILE_HEADER]++;
         vid_only_msg(NLS_PROCESSING_FILE_HEADER_N_OF, 1, nls_parms,
                      sizeof(nls_parms));
         page = i;
         sequence = 0;
         while (page != 0) {
            if ((erc = read_fhb(page)) == ercOK) {
               if (vlpb.display_structures)
                  display_fhb(&fhb, FALSE);
               if (fhb.fileHeaderNum == 0) {
                  unused_fhbs++;	/* Count now, verify later... */
                  page = 0;
               } else if (fhb.headerSequenceNum != 0 && sequence == 0) {
                  summary.count[TOTAL_FILES]--;
                  page = 0;
               } else {
                  if (sequence == 0) {
                     if (!vlpb.suppress_verify)
                        match_directory_entry_to_fhb();
                     if (vlpb.suppress_backup)
                        bypass_this_file = TRUE;
                     else {
                        bypass_this_file = (   fhb.fNoSave
                                            || (   fhb.modificationDT
                                                    <= vlpb.backup_after
                                                && fhb.creationDT
                                                    <= vlpb.backup_after));
                        if (bypass_this_file)
                           summary.count[BYPASSED]++;
                        else {
                           update_path();
                           memset(target.filespec, 0, sizeof(target.filespec));
                           BuildFileSpec(0, nls_parms[2].pb, &nls_parms[2].cb,
                                         NULL, last(target.filespec), FALSE,
                                         &path.node[1], path.node[0],
                                         &path.volume[1], path.volume[0],
                                         &path.directory[1], path.directory[0],
                                         &fhb.fileName[1], fhb.fileName[0],
                                         FALSE, NULL, 0, FALSE, 0);
                           log_msg(NLS_BACKING_UP, 0xFF01, nls_parms,
                                   sizeof(nls_parms));
                           target.lfa = 0;
                           target.size = find_file_length(page); /* m08 */
                           create_archive_sentinel(target.size);
/*  m08...                 target.size = (fhb.endOfFileLfa + (PAGE_SIZE - 1))
                                          & ~(PAGE_SIZE - 1);
...m08 */
                        }
                     }
                  }
                  for (j = 0; j < fhb.freeRunIndex ; j++) {
                     if (!vlpb.suppress_verify)
                        release_disk_run(fhb.vda[j], fhb.runLength[j]);
                     if (!bypass_this_file)
                        archive_disk_run(fhb.vda[j], fhb.runLength[j]);
                  }
                  if (     (page = fhb.extensionHeaderNumChain) == 0
                        && !bypass_this_file) {
                     summary.count[BACKED_UP_OK]++;
                     log_msg(NLS_DONE, 1, NULL, 0);
                  }
                  sequence++;
               }
            } else {
               if (sequence == 0) {
                  if (     erc != ercFileHeaderBadChecksum
                        && erc != ercFileHeaderBadPageNum
                        && fhb.fileHeaderNum != 0
                        && fhb.fileHeaderNum == fhb.fileHeaderPageNum)
                     summary.count[NOT_BACKED_UP]++;
               } else
                  log_msg(NLS_EXT_FHB_CHAIN_BROKEN, 1, NULL, 0);
               page = 0;
            }
         }
      }
   }
   summary.count[TOTAL_FILES] -= unused_fhbs - vhb.cFreeFileHeaders;
   if (!vlpb.suppress_backup)
      create_archive_EOD(final_close);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 CTOS volume structures provide for redundant file header information to
 increase the robustness of the disk structures after error.  This procedure
 attempts to read both the primary and the alternate file header and compare
 thhem to each other.  If there are errors in the primary file header, the
 alternate is used if it is error free---otherwise we're stuck with whatever
 condition the primary is in.  If the primary is error free, it is always used
 but any errors in the alternate are reported. */

unsigned read_fhb(unsigned fhb_num) {

   unsigned alt_erc, erc, page = fhb_num;;
   char buffer[PAGE_SIZE];
   sdType nls_parms[] = {(void *) &erc, sizeof(erc),
                         (void *) &alt_erc, sizeof(alt_erc),
                         (void *) &page, sizeof(page)};

   if ((erc = read_pages(buffer, vhb.lfaFileHeadersBase, page, 1)) != ercOK) {
      memset(&fhb, 0, sizeof(fhb));
      log_msg(NLS_PRIMARY_FHB_IO_ERROR, 1, nls_parms, sizeof(nls_parms));
   } else {
      memcpy(&fhb, buffer, sizeof(fhb));
      erc = valid_fhb(page, (FhbType *) buffer, FALSE);
   }
   if (vhb.altFileHeaderPageOffset == 0) {
      if (erc != ercOK) {
         filesys.invalid = TRUE;
         log_msg(NLS_NO_ALTERNATE_FHB, 1, nls_parms, sizeof(nls_parms));
      }
      return(erc);		/* Primary FHB is the only one we have */
   }
   if ((alt_erc = read_pages(buffer, vhb.lfaFileHeadersBase,
                             page + vhb.altFileHeaderPageOffset, 1)) != ercOK)
      log_msg(NLS_ALTERNATE_FHB_IO_ERROR, 1, nls_parms, sizeof(nls_parms));
   else 
      alt_erc = valid_fhb(page, (FhbType *) buffer, TRUE);
   if ((erc | alt_erc) == ercOK) {	/* Both FHB's valid: do they match? */
      if (_compare(&fhb, buffer, sizeof(fhb)) != 0) {
         filesys.invalid = TRUE;
         log_msg(NLS_FHB_DISAGREEMENT, 1, nls_parms, sizeof(nls_parms));
      }
      return(erc);			/* In any case, go with the primary */
   } else {
      filesys.invalid = TRUE;
      if (alt_erc == ercOK) {
         memcpy(&fhb, buffer, sizeof(fhb));	/* Alternate FHB is valid */
         return(alt_erc);
      } else
         return(erc);			/* Primary FHB is whatever it is */
   }

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 File Header validation has to check first that the checksum for the File
 Header equals zero.  If this is not true, not additional validation is
 performed.  Otherwise, consistency checks are made on different File Header
 fields contingent upon the class of file header: free, initial or extension.
 If any errors are found, a non-zero return code is provided to the user after
 the errors and the File Header data have been displayed. */

private unsigned valid_fhb(unsigned page, FhbType *fhb, Boolean alternate) {

   unsigned checksum = -MAGIC;
   enum {free_fhb, initial_fhb, extension_fhb} fhb_class;
   struct {
      Boolean checksum :1;
      Boolean page :1;
      Boolean sequence :1;
      Boolean filename :1;
      Boolean directory :1;
      Boolean extension :1;
      Boolean runs :1;
   } fhb_error;
   unsigned erc = ercOK, i;
   unsigned *raw_fhb = (unsigned *) fhb;

   *((Boolean *) &fhb_error) = FALSE;
   for (i = 0; i < sizeof(*fhb) / 2; i++)
      checksum += *raw_fhb++;
   if (checksum != 0)
      fhb_error.checksum = TRUE;
   else {
      if (fhb->fileHeaderPageNum != page)
         fhb_error.page = TRUE;
      if (fhb->fileHeaderNum == 0)
         fhb_class = free_fhb;
      else if (fhb->fileHeaderNum == fhb->fileHeaderPageNum)
         fhb_class = initial_fhb;
      else
         fhb_class = extension_fhb;
      switch (fhb_class) {
         case free_fhb:
            if (fhb->headerSequenceNum != 0)
               fhb_error.sequence = TRUE;
            if (fhb->fileName[0] != 0)
               fhb_error.filename = TRUE;
            if (fhb->dirName[0] != 0)
               fhb_error.directory = TRUE;
            break;

         case initial_fhb:
            if (fhb->headerSequenceNum != 0)
               fhb_error.sequence = TRUE;
            if (   fhb->fileName[0] == 0
                || fhb->fileName[0] > last(fhb->fileName))
               fhb_error.filename = TRUE;
            if (   fhb->dirName[0] == 0
                || fhb->dirName[0] > last(fhb->dirName))
               fhb_error.directory = TRUE;
            if (   fhb->extensionHeaderNumChain != 0
                && fhb->freeRunIndex != runsPerFhb)
               fhb_error.extension = TRUE;
            break;

         case extension_fhb:
            if (fhb->headerSequenceNum == 0)
               fhb_error.sequence = TRUE;
            if (fhb->freeRunIndex == 0)
               fhb_error.runs = TRUE;
            if (   fhb->extensionHeaderNumChain != 0
                && fhb->freeRunIndex != runsPerFhb)
               fhb_error.extension = TRUE;
            break;
      }
   }
   if (*((Boolean *) &fhb_error)) {
      sdType nls_parms[] = {(void *) &page, sizeof(page),
                            (void *) & fhb->fileHeaderPageNum,
                             sizeof(fhb->fileHeaderPageNum)};

      erc = ercFileHeaderInvalid;
      log_msg((alternate) ?
              NLS_ALTERNATE_FHB_INVALID : NLS_PRIMARY_FHB_INVALID,
              1, nls_parms, sizeof(nls_parms));
      if (fhb_error.checksum) {
         erc = ercFileHeaderBadChecksum;
         log_msg(NLS_CHECKSUM_ERROR, 1, NULL, 0);
      } else {
         if (fhb_error.page) {
            erc = ercFileHeaderBadPageNum;
            log_msg(NLS_FHB_PAGE_ERROR, 1, nls_parms, sizeof(nls_parms));
         }
         if (fhb_error.sequence) {
            erc = ercFileHeaderBadHeaderNum;
            log_msg(NLS_FHB_SEQ_ERROR, 1, NULL, 0);
         }
         if (fhb_error.filename) {
            erc = ercFileHeaderBadFileName;
            log_msg(NLS_FILENAME_ERROR, 1, NULL, 0);
         }
         if (fhb_error.directory)
            log_msg(NLS_DIRECTORY_ERROR, 1, NULL, 0);
         if (fhb_error.extension)
            log_msg(NLS_FHB_NOT_FULL, 1, NULL, 0);
         if (fhb_error.runs)
            log_msg(NLS_EXT_FHB_NO_RUNS, 1, NULL, 0);
      }
   }
   if (erc != ercOK)
      display_fhb(fhb, TRUE);
   return(erc);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 File header blocks are displayed to the user either when an internal
 consistency error has been detected or if the user has requested the verbose
 '[Display structures?]' option for Volume Archive.  In the first case, a
 hexadecimal dump of the entire file header is printed along with a brief
 summary of the file header contents. */

private void display_fhb(FhbType *fhb, Boolean hex) {


   static enum {free_fhb, initial_fhb, extension_fhb} fhb_class = 0xFF;
   char filespec[MAX_FILE_SPECIFICATION_LENGTH];
   unsigned i, nls_msg;
   sdType nls_parms[] = {(void *) &fhb->fileHeaderPageNum,
                                   sizeof(fhb->fileHeaderPageNum),
                         (void *) &fhb->fileHeaderNum,
                                   sizeof(fhb->fileHeaderNum),
                         (void *) &fhb->extensionHeaderNumChain,
                                   sizeof(fhb->extensionHeaderNumChain),
                         (void *) filespec, 0,
                         (void *) &fhb->endOfFileLfa,
                                   sizeof(fhb->endOfFileLfa),
                                  NULL, sizeof(fhb->vda[0]),
                                  NULL, sizeof(fhb->runLength[0]),
                         (void *) &i, sizeof(i)};

   if (fhb->fileHeaderNum == 0) {		/* Free... */
      if (fhb_class != free_fhb)
         log_msg(NLS_NEWLINE, 1, NULL, 0);	/* Pretty print */
      fhb_class = free_fhb;
      nls_msg = NLS_FREE_FHB_HEADER;
   } else if (fhb->fileHeaderNum == fhb->fileHeaderPageNum) {	/* ..initial */
      fhb_class = initial_fhb;
      memset(filespec, 0, sizeof(filespec));	/* Identify the file */
      BuildFileSpec(0, filespec, &nls_parms[3].cb, NULL, last(filespec),
                    FALSE, NULL, 0, NULL, 0, &fhb->dirName[1],
                    fhb->dirName[0], &fhb->fileName[1], fhb->fileName[0],
                    FALSE, NULL, 0, FALSE, 0);
      nls_msg = NLS_INITIAL_FHB_HEADER;
   } else {				/* ...or extension file header */
      fhb_class = extension_fhb;
      nls_msg = NLS_EXTENSION_FHB_HEADER;
   }
   log_msg(nls_msg, 1, nls_parms, sizeof(nls_parms));
   if (fhb_class == initial_fhb || fhb_class == extension_fhb) {
      for (i = 0; i < fhb->freeRunIndex; i++) {
         nls_parms[5].pb = (void *) &fhb->vda[i];
         nls_parms[6].pb = (void *) &fhb->runLength[i];
         log_msg(NLS_FHB_RUN, 1, nls_parms, sizeof(nls_parms));
      }
      log_msg(NLS_FHB_RUN_SUMMARY, 1, nls_parms, sizeof(nls_parms));
   }
   if (hex) {
      unsigned blackout_floor[] = {_offsetof(FhbType, password)};
      unsigned blackout_ceiling[] =
                          {_offsetof(FhbType, password) + last(fhb->password)};

      hex_dump(fhb, sizeof(*fhb), last(blackout_floor) + 1, blackout_floor,
               blackout_ceiling);
   }

}

pragma Page(1);
/* m08... */
/*-----------------------------------------------------------------------------
 This procedure finds the larger of fhb.endOfFileLfa or the summation
 of the disk runs lengths and return this.  This resolves the case where
 fhb.endOfFileLfa does not equal the disk run lengths - corrupt fileheader.
*/

unsigned long find_file_length (unsigned page) {

   unsigned j, sequence = 0, org_page, erc;
   unsigned long total_length = 0;
   sdType nls_parms[] = {(void *) &summary.count[CURRENT_FILE_HEADER],
                                  sizeof(summary.count[CURRENT_FILE_HEADER]),
                         (void *) &summary.count[TOTAL_FILE_HEADERS],
                                  sizeof(summary.count[TOTAL_FILE_HEADERS]),
                         (void *) &target.filespec[1], 0};

   org_page = page;
   while (page != 0) {
      if (fhb.fileHeaderNum == 0) {
         page = 0;
      } else if (fhb.headerSequenceNum != 0 && sequence == 0) {
         page = 0;
      } else {
         for (j = 0; j < fhb.freeRunIndex ; j++)
            total_length += fhb.runLength[j];
         page = fhb.extensionHeaderNumChain;
      }
      if ((page != 0) && (erc = read_fhb(page)) != ercOK)
         page = 0;
      sequence++;
   }
   erc = read_fhb(org_page);	/* this returns fhb to orginal state */
   return(total_length);
}

pragma Page(1);
/* ...m08 */
/*-----------------------------------------------------------------------------
 Each contiguous disk extent that belongs to a given file is read in the
 largest chunks possible (64 Kb less 512 bytes) and sent to the archive I/O
 co-process.  If an unrecoverable error is encountered while reading the file,
 report the VDA and skip over the (suspected) page in error. */

private void archive_disk_run(unsigned long vda, unsigned long run_length) {

   unsigned erc, xfer_size, xfer_count;
   iob_type *iob;
   Boolean io_error = FALSE, other_error = FALSE;
   sdType nls_parms[] = {(void *) &summary.count[CURRENT_FILE_HEADER],
                                  sizeof(summary.count[CURRENT_FILE_HEADER]),
                         (void *) &summary.count[TOTAL_FILE_HEADERS],
                                  sizeof(summary.count[TOTAL_FILE_HEADERS]),
                         (void *) &target.filespec[1], 0,
                         (void *) &erc, sizeof(erc),
                         (void *) &vda, sizeof(vda)};

   while (run_length > 0) {
      iob = allocate_iob(SEGMENT_SIZE - PAGE_SIZE);
      iob->ctrl = DATA;
      iob->data = iob->base;
      iob->available = 0;
      if (run_length > iob->size)
         xfer_size = iob->size;
      else
         xfer_size = run_length;
      do {
         erc = read_pages(iob->data, vda, 0, xfer_size / PAGE_SIZE);
         if (erc != ercOK)
            xfer_count = 0;
         else
            xfer_count = xfer_size;
         offset_of(iob->data) += xfer_count;
         iob->available += xfer_count;
         vda += xfer_count;
         run_length -= xfer_count;
         xfer_size -= xfer_count;
         if (erc == ercIoError)
            do {
               erc = read_pages(iob->data, vda, 0, 1);
               if (erc == ercIoError || erc == ercOK) {
                  if (erc == ercIoError) {
                     memset(iob->data, 0, PAGE_SIZE);
                     log_msg(NLS_FILE_IO_ERROR, 1, nls_parms,
                             sizeof(nls_parms));
                     io_error = TRUE;
                  }
                  offset_of(iob->data) += PAGE_SIZE;
                  iob->available += PAGE_SIZE;
                  vda += PAGE_SIZE;
                  run_length -= PAGE_SIZE;
                  xfer_size -= PAGE_SIZE;
               } else {
                  other_error = TRUE;
                  memset(iob->data, 0, xfer_size);
                  offset_of(iob->data) += xfer_size;
                  iob->available += xfer_size;
                  vda += xfer_size;
                  run_length -= xfer_size;
                  xfer_size = 0;
               }
            } while (erc == ercIoError && xfer_size > 0);
         else if (erc != ercOK) {
            other_error = TRUE;
            memset(iob->data, 0, xfer_size);
            offset_of(iob->data) += xfer_size;
            iob->available += xfer_size;
            vda += xfer_size;
            run_length -= xfer_size;
            xfer_size = 0;
         }
      } while (xfer_size > 0);
      Send(archive.msg_exch, iob);
   }
}

pragma Page(1);
/*-----------------------------------------------------------------------------
 All of the free file headers have (probably) been encountered once during the
 scan of all file headers on the volume.  However, the integrity of the links
 was not proven because we did not necessarily start at the beginning.  Test
 this integrity here by rereading them all... */

void verify_free_fhbs(void) {

   unsigned linked_free_fhbs = 0, i = vhb.iFreeFileHeader;

   log_msg(NLS_VERIFYING_FREE_FHBS, 1, NULL, 0);
   if (vlpb.display_structures && i != 0)
      log_msg(NLS_NEWLINE, 1, NULL, 0);
   while (   i != 0
          && linked_free_fhbs < vhb.cFreeFileHeaders
          && read_fhb(i) == ercOK) {
      if (vlpb.display_structures)
         display_fhb(&fhb, FALSE);
      linked_free_fhbs++;
      i = fhb.extensionHeaderNumChain;
   }
   if (     i != 0
         || linked_free_fhbs != vhb.cFreeFileHeaders
         || unused_fhbs != vhb.cFreeFileHeaders) {
      filesys.invalid = TRUE;
      log_msg(NLS_FREE_FHBS_BROKEN, 1, NULL, 0);
   }
}

/*-----------------------------------------------------------------------------
 Historically, a number of CTOS volume structures have been "hidden," that is,
 there is no entry in <Sys>FileHeaders.Sys that accounts for the space they
 occupy on the disk volume.  These structures are the initial and working
 Volume Home blocks, the allocation bit map and the directory indices for the
 individual directories (from some unknown, inconsistent reason the "master"
 directory of directories, <Sys>MFD.Sys, IS accounted for in the file
 headers.  In order for the bit map reconciliation to work, the space taken up
 by these structures must be released.  Note that the directories are released
 in another procedure during their own validation. */

void release_hidden_files(void) {

   release_disk_run(vhb.lfaInitialVhb, PAGE_SIZE);
   release_disk_run(vhb.lfaVhb, PAGE_SIZE);
   release_disk_run(vhb.lfaAllocBase,
                    vhb.allocPageCnt * (unsigned long) PAGE_SIZE);

}
