$MOD386
%Set(Debug,0)
%Set(TestClearSem,1)
%Set(CheckConsistency,0)
%Set(Log,0)
; If TestClearSem, then TimerInt code checks semaphores to see if they
; have been cleared.

; SemUtil.asm
; Utilities for CTOS Semaphore implementation
; 10/13/89 Jim Frandeen
; 05/30/91 Jim Frandeen: Remove test to see if the process that owns the
; 		semaphore is trying to lock it.
; 08/19/91 Add lock count to semaphore if user already owns it.
; 11/05/91 JM remove SemLockCritical, SemClearCritical
; 11/06/91 Jim Frandeen: make SemLock and SemClear fast enough for Denise
; 01/25/92 JM at ScInvalidHandle, handle error before LEAVE.
; 03/02/92 JF Implement Ram semaphores
; 04/02/92 JF Change SemClear logic so that another process can run before
;		we finish waking up waiters.  Implement temporary TestClearSem code
;		in TimerInt.
; 04/03/92 Move SemQuery here so we can query Ram semaphores.
; 06/10/92 JF & JF Put TestClearSem  code on a configurable switch.
;		:SuppressRamSemaphoreHack:  turns it off.
; 06/26/92 JF In MuxWait do Check with ints disabled -- otherwise we could
; 		end up with another message at the user's default exchange.
; 07/06/92 JF Turn off ram semaphore hack
; 08/24/92 JF Turn on ram semaphore hack
; 08/26/92 JF Turn off ram semaphore hack.  Put userNum in semWaitRec
;	so that termination will only remove wait recs of user terminating.
; 04/14/93 JM mask userNum slot bits in FindRamSemRec.

;**********************************************************************
; PUBLIC ENTRY PROCEDURES 
;**********************************************************************

PUBLIC	PidFromTssId
PUBLIC	SemAlloc
PUBLIC	SemAllocNode
PUBLIC	SemClear
PUBLIC	SemClearProcessLocks
PUBLIC	SemDeallocIfUnused
PUBLIC	SemDeallocNode
PUBLIC	SemLock
PUBLIC	SemMuxWait
PUBLIC	SemNotify
PUBLIC	SemQuery
PUBLIC	SemRestoreDS
PUBLIC	SemSaveDS
PUBLIC	SemSet
PUBLIC  SemTimerInt
PUBLIC	SemWait

;**********************************************************************
; LOCAL PROCEDURES
;**********************************************************************

PUBLIC AllocRamSemRec
PUBLIC CheckSemOptions
PUBLIC FindRamSemRec
PUBLIC SemAllocWaitSub
PUBLIC SemDeallocIfUnusedSub
PUBLIC SemError
PUBLIC SemWakeupSub
PUBLIC SemWaitForClearSub

;**********************************************************************
; EXTERNAL procedures
;**********************************************************************
EXTRN ChangePriority: FAR
EXTRN ChangeProcessPriority:FAR
EXTRN Check: FAR
EXTRN CheckErcBreak: FAR
EXTRN Crash: FAR
EXTRN KSuspendProcess: FAR
EXTRN SEND: FAR
EXTRN SemLockCritical: FAR
EXTRN SemClearCritical: FAR
EXTRN KUnSuspendProcess: FAR
EXTRN WAIT: FAR

;**********************************************************************
; Error codes
;**********************************************************************
ercNoSemMem				EQU 13950
ercSemLocked			EQU	13958
ercUserAlreadyOwnsLock	EQU 13959
ercSemOwnerKilled		EQU 13956
ercSemTimeout 			EQU 13955
ercSemInvalidHandle		EQU 13953
ercSemInvalidUser		EQU 13954

;**********************************************************************
; Literals
;**********************************************************************
sgZero					EQU 48H
rTss386LDT				EQU 96 
maskLdt 				EQU	4

;**********************************************************************
; DGroup definitions
;**********************************************************************
Data SEGMENT PUBLIC 'Data'
PUBLIC sgRamSemList
sgRamSemList DW 0
EXTRN nSemNodeFree :WORD
EXTRN oPcbRun :WORD
EXTRN pSemNodeFirst :DWORD
EXTRN pSemNodeLast :DWORD
EXTRN semAvail :DWORD
EXTRN semTRB :DWORD
sgSemNode EQU WORD PTR pSemNodeFirst+2
semTrbCounter			EQU WORD PTR semTRB
semTrbCounterReload		EQU WORD PTR semTRB+2
semTrbcEvents			EQU WORD PTR semTRB+4

PUBLIC cSem, cWaitingForClearSem
cSem					DD 0		;Placeholder
cWaitingForClearSem		DD 0

PUBLIC vfRamSemaphoreHack
vfRamSemaphoreHack		DB 0h		;default is false

%IF (%Debug) THEN (
PUBLIC cSemWaitCalled, cSemWait, cSemClear, cSemSet, cSemLock
PUBLIC cSemMuxWait, cSemMuxWaitCalled, cSemTimeout
PUBLIC cSemWaitForLock, cSemNotify, cSemClearProcessLocks, cSemQuery
PUBLIC cRamSemRecNotFound, cSemChangePriority, cSemChangePriorityBack
PUBLIC cSemClearRam, cSemWaitRamCalled, cSemWaitRam, cSemLockRam
PUBLIC cSemWaitForLockRam, cSemSetRam, cRamMuxSemRecNotFound
PUBLIC cSemQueryRam, cSemQueryRamLDT, cRestartWaitRecs
PUBLIC cNoSemMem
cSemWaitCalled 			DD 0
cSemWait 				DD 0			
cSemMuxWaitCalled 		DD 0			
cSemMuxWait 			DD 0			
cSemClear 				DD 0
cSemClearProcessLocks	DD 0
cSemSet 				DD 0
cSemLock 				DD 0
cSemWaitForLock 		DD 0
cSemTimeout 			DD 0
cSemNotify 				DD 0
cSemQuery 				DD 0
cSemQueryRam			DD 0
cSemQueryRamLDT			DD 0
cSemClearRam			DD 0
cSemWaitRamCalled		DD 0
cSemWaitRam				DD 0			
cSemLockRam				DD 0
cSemWaitForLockRam 		DD 0
cSemSetRam				DD 0
cRamSemRecNotFound		DD 0
cRamMuxSemRecNotFound	DD 0
cSemChangePriority 		DD 0		
cSemChangePriorityBack	DD 0		
cRestartWaitRecs		DD 0
cNoSemMem				DD 0
)FI
Data	ENDS
DGroup	GROUP Data

;WordToDword
;*************************************************************************
; The MOVZX doesn't work unless it has  an operand size override
; when moving a word to a doubleword.
;*************************************************************************
%*Define(WordToDword(to,from)) (
	DB		66h
	MOVZX	%to,%from
)
;**********************************************************************
; Semaphore record
; Note that a semaphore handle, passed to all semaphore procedures,
; points to the semLockFlag.  We must subtract semLockOffset
; to get a pointer to the first word of the semaphore record.
;**********************************************************************
sem					EQU  ES:[BX]
semLockOffset		EQU 12
sempNextUserOffset	EQU 40
sempNext			EQU DWORD PTR sem
sempPrev			EQU DWORD PTR sem+4
semHandle			EQU DWORD PTR sem+8
semLockFlag			EQU  WORD PTR sem+semLockOffset
semWaitingFlag		EQU  BYTE PTR semLockFlag+1
semOwnerTssId 		EQU  WORD PTR sem+14
semLockAndOwner		EQU DWORD PTR semLockFlag
semUserNum			EQU  WORD PTR sem+16
semLockCount		EQU  WORD PTR sem+18
semUserAndLockCount	EQU DWORD PTR sem+16
semOpenCount		EQU  WORD PTR sem+20
semPriority			EQU  BYTE PTR sem+22
semUseCount			EQU  BYTE PTR sem+23
semCountAndPriority	EQU DWORD PTR sem+20
semOptions			EQU  BYTE PTR sem+24
semFlags			EQU  BYTE PTR sem+25
semOptionsFlags		EQU  WORD PTR sem+24
sempFirstWaiting	EQU DWORD PTR sem+26
sempLastWaiting		EQU DWORD PTR sem+30
sempsbName			EQU DWORD PTR sem+34
semOpenCountOwner	EQU  WORD PTR sem+38
sempNextUser		EQU DWORD PTR sem+40
sempLastUser		EQU DWORD PTR sem+44
semFlagOwnerKilled	EQU 1
semFlagRamSem		EQU 2
semFlagInUse_mask	EQU 4
semFlagInUse_bit	EQU 2
semOpCritical		EQU 1
semOpNonTerm		EQU 2
semOpExclusive		EQU 4

pNext	EQU DWORD PTR 0
raNext	EQU 0
saNext	EQU 2
pPrev	EQU DWORD PTR 4
raPrev	EQU 4
saPrev	EQU 6

;**********************************************************************
; SetSemWaiting
; ES:BX points to sem record
; Set the bit in the high byte of the ram semaphore
; that indicates a process is waiting
;**********************************************************************
%*DEFINE (SetSemWaiting) (
	OR		semWaitingFlag,1
)
;**********************************************************************
; SetRamSemWaiting
; ES:BX points to semaphore
; Set the bit in the high byte that indicates a process is waiting
;**********************************************************************
%*DEFINE (SetRamSemWaiting) (
	OR		WORD PTR ES:[BX],100H
)
;**********************************************************************
; QueueSem
; ES:BX points to sem to queue
; GS:DI points to prev entry or list head if none.
; Uses GS,DI,EAX
;**********************************************************************
%*DEFINE (QueueSem) (
	MOV		EAX,GS:pNext[DI]	;EAX points to next entry or list head
	MOV		ES:pNext[BX],EAX	;set next pointer of new node
	MOV		ES:raPrev[BX],DI	;set prev pointer of new node
	MOV		ES:saPrev[BX],GS	;to point to current entry
	MOV		GS:raNext[DI],BX	;next pointer of current entry 
	MOV		GS:saNext[DI],ES	;points to new node
	LGS		DI,ES:pNext[BX]		;GS:DI points to next entry or list head
	MOV		GS:raPrev[DI],BX	;prev pointer of next entry points to 
	MOV		GS:saPrev[DI],ES	;new node
)
;**********************************************************************
; DeQueueSem
; ES:BX points to sem to dequeue
; Uses GS,DI,EAX
;**********************************************************************
%*DEFINE (DeQueueSem) (
	LGS		DI,ES:pPrev[BX]		;GS:DI points to prev entry or list head 
	MOV		EAX,ES:pNext[BX]
	MOV		GS:pNext[DI],EAX	;Move next pointer to prev node
	LGS		DI,ES:pNext[BX]		;GS:DI points to next entry or list head 
	MOV		EAX,ES:pPrev[BX]
	MOV		GS:pPrev[DI],EAX	;Move prev pointer to next node
)
;**********************************************************************
; DeallocSem
; ES:BX points to sem to deallocate
; Uses GS,DI,EAX
;**********************************************************************
%*DEFINE (DeallocSem) LOCAL InUseOK OpenOK LockOK UseOK(
%IF(%CheckConsistency) THEN (
	PUSH	0FFh				;Deallocate
	PUSH	BX					;oSemRec
	CALL	ValidateSemNodeFree
	XOR		AX,AX
	CMP		semOpenCount,AX
	JE		%OpenOK
	INT		3
%OpenOK:
	CMP		semLockCount,AX
	JE		%LockOK
	INT		3
%LockOK:
	CMP		semUseCount,AL
	JE		%UseOK
	INT		3
%UseOK:
; Make sure semRec is in use.
	BTR		semFlags,semFlagInUse_bit
	JC		%InUseOK
	INT		3
%InUseOK:
)FI
	INC		nSemNodeFree
	LGS		DI,semAvail+4		;GS:DI points to last node in list
	%QueueSem					;Queue to free list
)
;**********************************************************************
; Semaphore Wait record
; Note that FS:[SI] points to this record for all references.
;**********************************************************************
semWaitRec				EQU  FS:[SI]
semWaitpNext			EQU DWORD PTR semWaitRec
semWaitpPrev			EQU DWORD PTR semWaitRec+4
semWaitTssId 			EQU  WORD PTR semWaitRec+8
semWaitUserNum			EQU  WORD PTR semWaitRec+10
semWaitExch				EQU  WORD PTR semWaitRec+12
semWaitTimeout			EQU  WORD PTR semWaitRec+14
semWaitFlags			EQU  WORD PTR semWaitRec+16

semWaitFlagNotify_mask	EQU 1
semWaitFlagWokeUp_mask	EQU 2
semWaitFlagWokeUp_bit	EQU 1
semWaitFlagMux_mask		EQU 4
semWaitFlagInUse_mask	EQU 8
semWaitFlagInUse_bit	EQU 3

;**********************************************************************
; QueueSemWaitRec
; FS:SI points to semWaitRec to queue
; GS:DI points to prev entry or list head if none.
; Uses GS,DI,EAX
;**********************************************************************
%*DEFINE (QueueSemWaitRec) (
	MOV		EAX,GS:pNext[DI]	;EAX points to next entry or list head
	MOV		FS:pNext[SI],EAX	;set next pointer of new node
	MOV		FS:raPrev[SI],DI	;set prev pointer of new node
	MOV		FS:saPrev[SI],GS	;to point to current entry
	MOV		GS:raNext[DI],SI	;next pointer of current entry 
	MOV		GS:saNext[DI],FS	;points to new node
	LGS		DI,FS:pNext[SI]		;GS:DI points to next entry or list head
	MOV		GS:raPrev[DI],SI	;prev pointer of next entry points to 
	MOV		GS:saPrev[DI],FS	;new node
)
;**********************************************************************
; DeQueueSemWaitRec
; FS:SI points to semWaitRec to dequeue
; Uses GS,DI,EAX
;**********************************************************************
%*DEFINE (DeQueueSemWaitRec) (
	LGS		DI,FS:pPrev[SI]		;GS:DI points to prev entry or list head 
	MOV		EAX,FS:pNext[SI]
	MOV		GS:pNext[DI],EAX	;Move next pointer to prev node
	LGS		DI,FS:pNext[SI]		;GS:DI points to next entry or list head 
	MOV		EAX,FS:pPrev[SI]
	MOV		GS:pPrev[DI],EAX	;Move prev pointer to next node
)
;**********************************************************************
; DeallocSemWaitRec
; FS:SI points to semWaitRec to deallocate
; Uses GS,DI,EAX
; WaitRec is deallocated by:
;	(1) MuxWait.  MuxWait allocates a waitRec for each semaphore it
;		is waiting for.  When it wakes up, it deletes the waitRecs
;		it created.
;	(2)	SemWaitForClearSub.  It is called with a pointer to a
;		waitRec, and it deletes that waitRec when it wakes up.
;	(3) SemWakeup deletes waitRec for SemNotify.
;**********************************************************************
%*DEFINE (DeallocSemWaitRec) LOCAL WaitInUseOk(
%IF(%CheckConsistency) THEN (
	PUSH	0FFh				;Deallocate
	PUSH	SI					;oSemRec
	CALL	ValidateSemNodeFree
	BTR		semWaitFlags,semWaitFlagInUse_bit
	JC		%WaitInUseOk
	INT		3	
%WaitInUseOk:
)FI
	INC		nSemNodeFree
	LGS		DI,semAvail+4		;GS:DI points to last node in list
	%QueueSemWaitRec			;Queue to free list
)
;**********************************************************************
; Semaphore MuxWait record
;**********************************************************************
muxSemRec				EQU  ES:[BX]
muxSemCount				EQU  WORD PTR muxSemRec
; muxSemCount has number of muxSemList entries
muxSemList				EQU  DWORD PTR muxSemRec+2
; Each muxSemList entry has one reserved word and one sh
muxSemReserved			EQU   WORD PTR muxSemRec+2
muxSemSh				EQU  DWORD PTR muxSemRec+4
muxSemListSize			EQU  6
;**********************************************************************
; Error macro
;**********************************************************************
%*Define(Error(erc)) (
	MOV		AX,%erc
	CALL	SemError
)
;**********************************************************************
; TestSh(RamSemaphoreLabel, InvalidHandleLabel, CheckOptionsLabel)
; sh is semaphore handle to load.
; On return, ES:BX points to system semaphore record or ram semaphore.
; CX and ES have DGroup.
; Note that a system semaphore handle points to the semLockFlag.  
; We must subtract semLockOffset
; to get a pointer to the first word of the semaphore record.
; If Ram Semaphore, jump to RamSemaphoreLabel.
; If system semaphore is invalid, jump to InvalidHandleLabel.
;**********************************************************************
%*DEFINE (TestSh(RamSemaphoreLabel, InvalidHandleLabel, CheckOptionsLabel)) (
	MOV		CX,DGroup
	MOV		ES,CX
	MOV		BX,sh+2
	CMP		BX,ES:sgSemNode			;Is this a system semaphore handle?
	MOV		ES,BX					;ES = sg of handle
	MOV		BX,sh					;ES:BX = sh
	JNE		%RamSemaphoreLabel		;Must be a Ram semaphore
; Verify that the semHandle in the semaphore record is the same as
; the handle passed in.
	CMP		BX,semHandle-semLockOffset
	JNE		%InvalidHandleLabel
	SUB		BX,semLockOffset		;ES:BX points to system sem rec
; We have a valid system semaphore handle in ES:BX
	CMP		semOptions,0
	JNE		%CheckOptionsLabel
)

;**********************************************************************
; IncCounter macro
;**********************************************************************
%*Define(IncCounter(counter)) (
%IF (%Debug) THEN (
	PUSH	DS
	PUSH	DGroup			;So as not to use a register
	POP		DS
	INC		%counter
	POP		DS
)FI
)
;**********************************************************************
; CallDebugger macro
;**********************************************************************
%*Define(CallDebugger) (
%IF (%Debug) THEN (
	INT		3
)FI
)
;**********************************************************************
; PCB definitions
;**********************************************************************
pcbPriority			EQU 3
pcbExchSync			EQU 12
pcbUserNum			EQU	14
pcbTssId			EQU 6

SemUtilCode	SEGMENT PUBLIC 'Code'
ASSUME	CS:SemUtilCode, DS:DGroup, SS:NOTHING

AllocRamSemRec PROC NEAR
;**********************************************************************
; AllocRamSemRec(sh)
; Search the list of Ram semaphores to see if we
; have a record for it.  If we don't already have one, allocate one
; and initialize it.
; Enter and exit with interrupts disabled.
; On exit, ES:BX points to Ram semaphore record.
; Erc is in AX and condition code is set
;**********************************************************************
sh			EQU DWORD PTR [BP+4]
	ENTER	0,0
	MOV		EAX,sh
	CALL	FindRamSemRec
	JZ		ArRetOK					;ES:BX points to Ram semRec

; ES:BX points to the head of the list for this user if the list is empty,
; or to the pNextUser field of the last Ram semaphore for this user.
	PUSH	ES						;Save pointer to prev ram SemRec
	PUSH	BX						;for this user.
; Allocate a semaphore record and Queue the Ram semRec to the list of 
; semaphores in the system so that SemTimerInt will find it.  
	PUSH	pSemNodeLast
	CALL	FAR PTR SemAlloc		;(pSemPtr)
; Queue the ram SemRec to the list of Ram semaphores for this user.
; We do this so that we can find a Ram semaphore faster.  We only have to 
; look in the list for the user, rather than the list of all semaphores
; in the system.  Note that this semRec is on two queues:
; the list of Ram semaphores for the user and the list of semaphores
; in the system.
	POP		DI						;GS:DI points to prev Ram semrec
	POP		GS						; for user or list head
	JNZ		ArRet					;AX = ercNoSemMem
	ADD		BX,sempNextUserOffset	;Queue to pNextUser field
	%QueueSem
	SUB		BX,sempNextUserOffset	;BX points to first byte of record
; Initialize the rest of the fields in the Ram semaphore record.
	MOV		SI,oPcbRun				;SI points to our pcb
	MOV		AX,[SI+pcbUserNum]		;AX has our usernum
	MOV		semUserNum,AX
	MOV		semFlags,semFlagRamSem
	MOV		EAX,sh
	MOV		semHandle,EAX
%IF(%CheckConsistency) THEN (
	BTS		semFlags,semFlagInUse_bit
	JNC		ArInUseOK
	%CallDebugger
ArInUseOK:
)FI
ArRetOK:
	XOR		AX,AX					;Erc = ercOK
ArRet:
	LEAVE
	RET		4
AllocRamSemRec ENDP


CheckSemOptions PROC NEAR
;**********************************************************************
; On entry, ES:BX points to system SemRec.
; CX has DGroup.
; Check for valid options.
; Only SemLockCritical and SemClearCritical specify a critical section 
; semaphore.
; On return, condition code is set.
; Uses FS, SI, AX
;**********************************************************************
	TEST	semOptions,semOpCritical
	JNZ		CoInvalidHandle
	TEST	semOptions,semOpExclusive
	JZ		CoRetOK
; If this semaphore was opened for exclusive use, make sure it
; belongs to the user that is running.
	MOV		FS,CX					;FS has DGroup
	MOV		SI,FS:oPcbRun			;SI points to our pcb
	MOV		SI,FS:[SI+pcbUserNum]	;SI has our usernum
	CMP		SI,semUserNum
	JNE		CoInvalidUser
CoRetOK:
	XOR		AX,AX					;Condition code = 0
	RET

CoInvalidUser:
	%Error(ercSemInvalidUser)
	OR		AX,AX					;Set condition code
	RET
CoInvalidHandle:
	%Error(ercSemInvalidHandle)
	OR		AX,AX					;Set condition code
	RET
CheckSemOptions ENDP	
FindRamSemRec PROC NEAR
;**********************************************************************
; Search the list of Ram semaphores for this user to see if we
; have a record for it.
; Interrupts are disabled.
; On entry, EAX has semaphore handle.
; On exit, if found, condition code = 0 and
;	ES:BX points to semRec 
; On exit, if not found, condition code <> 0 and
; 	ES:BX points to the head of the list if the list was empty, or
;	to the sempNextUser field of the last record in the list.
;**********************************************************************
	MOV		BX,oPcbRun				;BX points to our pcb
	MOV		BX,[BX+pcbUserNum]		;BX has our usernum
	AND		BX, 3ffh				;mask slot bits, 04/14/93 JM
	SHL		BX,3					;*8
	MOV		ES,sgRamSemList			;ES:BX points to user ramSemList
	MOV		CX,BX					;CX:DX has sg:ra of list head
	MOV		DX,ES
FrTestLast:
; Check to see if the list is empty for this user.
	CMP		CX,ES:[BX]
	JNE		FrNext
	CMP		DX,ES:[BX+2]
	JNE		FrNext
; The list is empty, or the list does not contain the specified sh.  
; If empty, ES:BX points to list head.
; If sh is not in the list, ES:BX points to last node in list.
; The semaphore could belong to another user, so we must search
; the list of all semaphores.
	PUSH	ES						;Save pointer to end of list
	PUSH	BX
	LES		BX,pSemNodeFirst
	LEA		CX,pSemNodeFirst
	CMP		BX,CX
	JE		FrTestEmpty
FrTestFirstHandle:
	CMP		EAX,semHandle
	JE		FrFound
	MOV		DX,sempNext
FrTestLastSys:
	CMP		DX,CX
	JE		FrLast
FrTestHandle:
; Load offset of next semaphore -- sg is always the same
; except for the last record, that points to the head of the list.
	MOV		BX,DX
	CMP		EAX,semHandle
	JE		FrFound
	MOV		DX,sempNext
	CMP		DX,CX
	JE		FrLast
; Load offset of next semaphore -- sg is always the same...
	MOV		BX,DX
	CMP		EAX,semHandle
	JE		FrFound
	MOV		DX,sempNext
	CMP		DX,CX
	JE		FrLast
; Load offset of next semaphore -- sg is always the same...
	MOV		BX,DX
	CMP		EAX,semHandle
	JE		FrFound
	MOV		DX,sempNext
	JMP		FrTestLastSys

FrTestEmpty:
; If the ra of the listhead matches the ra of the list head, 
; the list is probably empty.  It is possible that the ra of the list head
; is the same as the ra of a semNode.
	CMP		WORD PTR pSemNodeFirst+2,DGroup
	JNE		FrTestFirstHandle
	JMP		SHORT FrEndList

FrLast:
; If the ra of the pointer matches the ra of the list head, we are
; probably at the end of the list.  It is possible that the ra of the list head
; is the same as the ra of a semNode.
	CMP		WORD PTR sempNext+2,DGroup
	JNE		FrTestHandle
FrEndList:
; Restore ES:BX to point to list head or last node in list.
	POP		BX					
	POP		ES
	OR		AX,1					;Condition code <> 0
	RET
FrFound:
	POP		EAX						;Discard pointer to list head
	XOR		AX,AX					;Condition code = 0
	RET

FrNext:
	LES		BX,DWORD PTR ES:[BX]
; ES:BX points to sempNextUser field of semRec.
	CMP		EAX,semHandle-sempNextUserOffset
	JNE		FrTestLast
; ES:BX points to sempNextUser field of semRec that has matching handle.
	SUB		BX,sempNextUserOffset
; ES:BX points to semRec
	XOR		AX,AX					; Condition code = 0
	RET
FindRamSemRec ENDP	

PidFromTssId PROC FAR
;**********************************************************************
; PidFromTssId: PROCEDURE(tssId) WORD
; Given a TSS, return oPcb of the owner of the tss
;**********************************************************************
tssId		EQU WORD PTR [BP+6]
tss_oPcb		EQU 166		;0A6H
	ENTER	0,0
; get data alias for TSS
	MOV		AX,tssId
	SUB		AX,8 
 	MOV		ES,AX
	MOV		AX,ES:[tss_oPcb]
	LEAVE	
	RET		2
PidFromTssId ENDP

SemAlloc PROC FAR
;**********************************************************************
; SemAlloc(pSemPrev) pNewNode
; Allocate a semaphore record and chain it to pSemPrev.
; On return, ES:BX points to new semaphore record, and record
; has been initialized.
; Erc is in AX and condition code is set.
; Erc is not used if called from PLM
; If no nodes are available, ES:BX = 0 
;**********************************************************************
pSemPrev	EQU DWORD PTR [BP+6]
	ENTER	0,0
	PUSH	pSemPrev
	CALL	FAR PTR SemAllocNode
	JNZ		SalRet					;ES:BX = 0
; ES:BX points to new semaphore node.
	XOR		EAX,EAX
	MOV		semLockAndOwner,EAX
	MOV		semUserAndLockCount,EAX
	MOV		semCountAndPriority,EAX
	MOV		semOptionsFlags,AX
	MOV		sempsbName,EAX
	MOV		semOpenCountOwner,AX
; Initialize list of waiting records to nil.
	LEA		AX,sempFirstWaiting
	MOV		sempFirstWaiting,AX
	MOV		sempLastWaiting,AX
	MOV		sempFirstWaiting+2,ES
	MOV		sempLastWaiting+2,ES
; Initialize list of users who have opend semaphore to nil.
	LEA		AX,sempNextUser
	MOV		sempNextUser,AX
	MOV		sempLastUser,AX
	MOV		sempNextUser+2,ES
	MOV		sempLastUser+2,ES
	XOR		AX,AX					;erc = ercOK
%IF(%CheckConsistency) THEN (
	BTS		semFlags,semFlagInUse_bit
	JNC		SalInUseOK
	%CallDebugger
SalInUseOK:
)FI
SalRet:
	LEAVE
	RET		4
SemAlloc ENDP

SemAllocNode PROC FAR
;**********************************************************************
; SemAllocNode(pSemPrev) pNewSemNode
; Interrupts are disabled.
; Allocate a semaphore record and chain it to pSemPrev
; if pSemPrev <> 0.
; On return, ES:BX points to new semaphore node if available,
; otherwise, ES:BX is zero.
; On return, erc is in AX and condition code is set.
;**********************************************************************
pSemPrev	EQU DWORD PTR [BP+6]
	ENTER	0,0
	CALL	SemAllocSub
	JZ		SanGotNode			;AX = ercOK
	XOR		BX,BX
	MOV		ES,BX
	OR		AX,AX				;Set condition code
	JMP		SHORT SanRet		;AX has erc
SanGotNode:
; ES:BX points to new semaphore node.
	CMP		pSemPrev,AX			;pSemPrev = 0?
	JE		SanRet				;Yes, nothing to queue
	LGS		DI,pSemPrev
	%QueueSem
	XOR		AX,AX				;erc = ercOK
SanRet:
	LEAVE	
	RET		4
SemAllocNode ENDP

SemDeallocNode PROC FAR
;**********************************************************************
; SemDeallocNode(pSemNode, fUnchain)
; Deallocate a semaphore record.  If fUnchain, remove the semNode from
; the list that it is chained to.
;**********************************************************************
pSemNode	EQU DWORD PTR [BP+8]
fUnchain	EQU  BYTE PTR [BP+6]
	ENTER	0,0
	LES		BX,pSemNode
	PUSHF
	CLI								;DISABLE
	TEST	fUnchain,0FFh
	JZ		SdDealloc
	%DeQueueSem
SdDealloc:
%IF(%CheckConsistency) THEN (
	OR		semFlags,semFlagInUse_mask
	XOR		AX,AX
	MOV		semOpenCount,AX
	MOV		semLockCount,AX
	MOV		semUseCount,AL
)FI
	%DeallocSem
	POPF							;ENABLE
	LEAVE
	RET		6
SemDeallocNode ENDP

SemAllocSub PROC NEAR
;**********************************************************************
; Allocate a semaphore record.
; Interrupts are disabled.
; On return, erc is in AX and condition code is set.
; 	ES:BX points to semRec
;**********************************************************************
	DEC		nSemNodeFree
	JS		SaNoneFree
; Unchain node from free list
	LES		BX,semAvail				;ES:BX points to free node
	%DeQueueSem
SaRetOk:
	XOR		AX,AX					;erc = ercOK
SaRet:
	RET		

SaNoneFree:
; Come here if there are no free nodes
	INC		nSemNodeFree		;Restore count to zero
	%Error(ercNoSemMem)
	OR		AX,AX				;Set condition code erc <> ercOK
	RET
SemAllocSub ENDP
SemAllocWaitSub PROC NEAR
;**********************************************************************
; Allocate wait record and chain it to the semaphore record.
; Start the trb if it is not already running.
; On entry, ES:BX points to semRec
; 	CX has timeout value
; 	Interrupts are disabled.
; On return, erc is in AX and condition code is set.
; 	FS:SI points to waitRec.
; If no wait record can be allocated, the semRec is deallocated if
; it is unused.
;**********************************************************************
	DEC		nSemNodeFree
	JS		SwNoneFree
; Unchain node from free list
	LFS		SI,semAvail				;FS:SI points to free node
	%DeQueueSemWaitRec
; Chain new wait record onto semaphore record.
	LGS		DI,sempLastWaiting		;GS:DI points end of queue
	%QueueSemWaitRec
; Initialize fields in semaphore wait record.
	STR  	AX						;AX = our TssId
	MOV		semWaitTssId,AX
	MOV		DI,oPcbRun				;DI points to our pcb
	MOV		AX,pcbUserNum[DI]
	MOV		semWaitUserNum,AX
	MOV		AX,pcbExchSync[DI]		;AX = our exchange
	MOV		semWaitExch,AX
	MOV		semWaitFlags,0
%IF(%CheckConsistency) THEN (
	OR		semWaitFlags,semWaitFlagInUse_mask
)FI
	MOV		semWaitTimeout,CX		;CX has timeout value
	CMP		CX,0FFFFh
	JE		SwRetOk					;No timeout
;	Check to see if the current timeout value is less than the
;	current value left in the counter.  If so, set the current
;	value of the counter to go off sooner.
	CMP		CX,semTrbCounter
	JAE		SwCheckReload
	MOV		semTrbCounter,CX
SwCheckReload:
	CMP		CX,semTrbCounterReload
	JAE		SwInitEvents
	MOV		semTrbCounterReload,CX
SwInitEvents:
;	Set cEvents to zero to make sure the clock is running.
;	The clock will not run if there are no timeouts pending.
	MOV		semTrbcEvents,0
SwRetOk:
	XOR		AX,AX					;erc = ercOK
	RET		

SwNoneFree:
; Come here if there are no free nodes
	INC		nSemNodeFree		;Restore count to zero
; Delete the semaphore record if it is not being used.
	CALL	SemDeallocIfUnusedSub
	%Error(ercNoSemMem)
	OR		AX,AX				;Set condition code erc <> ercOK
	RET
SemAllocWaitSub ENDP

%IF(0) THEN (
; No longer used, but save the code for possible future use.
SemChain PROC FAR
;**********************************************************************
; SemChain: PROCEDURE(pNewNode, pCurrentNode)
; Chain entry onto list.
; On entry, BX points to entry to chain, and 
; DI points to prev entry or list head if none.
; On return, SI points to next entry or list head if none.
;**********************************************************************
	ENTER 0,0
	PUSH DS
	LES  BX,DWORD PTR[BP+10]	;ES:BX points to new node
	LDS  DI,DWORD PTR[BP+6]		;DS:DI points to current node
	MOV  SI,raNext[DI]			;AX:SI points to next entry or list head
	MOV  AX,saNext[DI]
	MOV  ES:raNext[BX],SI		;set next pointer of new node
	MOV  ES:saNext[BX],AX
	MOV  ES:raPrev[BX],DI		;set prev pointer of new node
	MOV  ES:saPrev[BX],DS		;to point to current entry
	MOV  raNext[DI],BX			;next pointer of current entry 
	MOV  saNext[DI],ES			;points to new node
	MOV  DS,AX
	MOV  raPrev[SI],BX			;prev pointer of next entry points to 
	MOV  saPrev[SI],ES			;new node
	POP  DS
	LEAVE
	RET  8
SemChain ENDP
)FI
SemClear PROC FAR
;**********************************************************************
; SemClear: PROCEDURE(sh) ErcType PUBLIC REENTRANT; 
;**********************************************************************
sh				EQU DWORD PTR [BP+6]
pSemRec			EQU DWORD PTR [BP-2*2]
raSemRec		EQU  WORD PTR pSemRec
sgSemRec		EQU  WORD PTR pSemRec+2
DsSave			EQU  WORD PTR [BP-2*3]
	ENTER	2*3,0
	%TestSh(ScRamSemaphore, ScInvalidHandle, ScsCheckOptions)
ScsOptionsValid:
; ES:BX now points to valid system semRec
	%IncCounter (cSemClear)
	XOR		AX,AX					;erc = ercOK
	PUSHF
	CLI								;DISABLE
; Note that the user can call SemClear when the lockCount is already zero,
; so check for this.
	CMP		semLockCount,AX			;lockCount = 0 already?
	JE		ScsUnlock				;Yes, don't decrement
	DEC		semLockCount				;Decrement use count
; If the use count is not zero, then return without waking up any waiters.
; Note that SemClear did not work like this before conversion to .asm --
; it used to always wake up waiters.
	JNZ		ScsQuickRet
ScsUnlock:
	MOV		semLockFlag,AX			;Zero lock
; Check to see if there is a process waiting.
	LEA		DX,sempFirstWaiting
	CMP		DX,sempFirstWaiting
	JNE		ScsWaiting
; Check to see if the open count is zero.  
; If so, it requires  special handling.
	CMP		semOpenCount,AX
	JE		ScsOpenCountZero
; No processes are waiting, so easy out.

ScsQuickRet:
	POPF							;ENABLE
	LEAVE
	RET		4

ScsOpenCountZero:
; Open count is zero and no processes are waiting for this semaphore.
; Maybe we can deallocate.
	MOV		DsSave,DS				;Save DS
	MOV		DS,CX					;CX = DGroup
	CALL	SemDeallocIfUnusedSub
	JMP		ScRet

ScsCheckOptions:
; Come here if the options in the semaphore record are anything
; other than zero.
	CALL	CheckSemOptions
	JZ		ScsOptionsValid
	JMP		ScErrorRet

ScsWaiting:
; Come here if the semaphore is clear and
; we have at least one process waiting for the semaphore.
; We must wake up all waiting processes.
; CX = DGroup
	MOV		DsSave,DS				;Save DS
	MOV		DS,CX					;DS = DGroup

ScScanWaitList:
; Common code for Ram and system semaphores
	MOV		raSemRec,BX				;Save pointer to semRec
	MOV		sgSemRec,ES
; Increment the use count so that the record will not be deleted
; until we are finished with it.
	INC		semUseCount
%IF(%CheckConsistency) THEN (
; Make sure semRec is in use.
	TEST	semFlags,semFlagInUse_mask
	JNZ		ScInUseOK
	INT		3
ScInUseOK:
)FI

ScHandleNextWaitRec:
	LFS		SI,sempFirstWaiting
ScTestLastWaitRec:
	LGS		DI,sh
	LEA		DX,sempFirstWaiting
	CMP		SI,DX
	JE		ScWakeupsFinished
; There is another process waiting.
; Check to see if the semaphore is still clear.  If not, don't 
; continue waking up waiters -- it could be an endless loop if every time
; we wake up a process, that process runs, the semaphore gets set again,
; and someone else waits for it.
	TEST	WORD PTR GS:[DI],1
	JNZ		ScSemSetAgain
ScTestWokeUp:
; Check to see if we have already woken up the waiting process.
	TEST	semWaitFlags,semWaitFlagWokeUp_mask
	JZ		ScWakeupWaiter
; We already woke up this process -- try the next.
	MOV		SI,semWaitpNext
	CMP		SI,DX
	JE		ScWakeupsFinished
	JMP		ScTestWokeUp

ScWakeupWaiter:
; Wake up next waiting process. 
; NOTE: the other process may run.  We must leave things in a state
; so that, if another process runs, things will be in a consistent state.
	CALL	SemWakeupSub			;ES:BX, FS:SI preserved
%IF(%CheckConsistency) THEN (
; Make sure semRec is in use.
	TEST	semFlags,semFlagInUse_mask
	JNZ		ScInUseOK1
	%CallDebugger
ScInUseOK1:
)FI
; Note that the WaitRec may have been deleted if another process ran.
; We must start at the beginning of the list every time.
	POPF							;ENABLE
	PUSHF
	CLI
	JMP		ScHandleNextWaitRec

ScSemSetAgain:
; Set the high byte of the Ram semaphore to indicate a process
; is waiting for it.
	OR		WORD PTR GS:[DI],100H

ScWakeupsFinished:
; When we have woken up all waiting processes, semPriority will be <> 0
; if our priority was changed by a process that wanted the lock.
	CMP		semPriority,0
	JE		ScDealloc
; Our priority was raised by a process that wanted the lock so 
; that we would hurry up and finish. Change it back to where it was.
; Note that lowering our priority will cause the other process to run.
	PUSH	WORD PTR semPriority
	CALL	ChangePriority
	LES		BX,pSemRec				;Restore pointer to semRec
	%IncCounter (cSemChangePriorityBack)
ScDealloc:
; Decrement the use count for this semaphore.
	DEC		semUseCount
	CALL	SemDeallocIfUnusedSub
	JMP		ScRet

ScRamSemaphore:
;**********************************************************************
; If the sg of the semaphore handle does not match sgSemNode, then
; we assume it's a Ram semaphore.
; ES:BX points to Ram semaphore.
;**********************************************************************
	%IncCounter (cSemClearRam)
	XOR		EAX,EAX
	PUSHF
	CLI								;DISABLE
	XCHG	EAX,ES:[BX]				;AX has previous value of semaphore
; If byte 1 of the semaphore is 1, then there is a process waiting
; for the semaphore.
	OR		AH,AH
	JNZ		ScrWaiting
	POPF							;ENABLE
	XOR		AX,AX					;erc = ercOK
	LEAVE
	RET		4

ScrWaiting:
	MOV		DsSave,DS				;Save DS
	MOV		DS,CX					;DS = DGroup
	MOV		EAX,sh
	CALL	FindRamSemRec
	JNZ		ScrSemRecNotFound
; ES:BX points to semRec of ram semaphore
; Check to see if there is a process waiting.
	LEA		DX,sempFirstWaiting
	CMP		DX,sempFirstWaiting
	JNE		ScScanWaitList
; If there is no semaphore wait record, something is wrong.
	CALL	SemDeallocIfUnusedSub
	%CallDebugger	

ScrSemRecNotFound:
; We did not find a semaphore record for this Ram semaphore.
; If the semaphore timed out, the wait record would be deleted,
; and the semaphore record would be deleted.
	%IncCounter (cRamSemRecNotFound)

ScRet:
	MOV		DS,DsSave					;Restore DS
	POPF								;ENABLE
	XOR		AX,AX						;Erc = ercOk
	LEAVE
	RET		4

ScInvalidHandle:
	%Error(ercSemInvalidHandle)
ScErrorRet:
	LEAVE
	RET		4
SemClear ENDP

SemClearProcessLocks PROC FAR
;**********************************************************************
; SemClearProcessLocks: PROCEDURE(pid) ErcType PUBLIC REENTRANT; 
; Clear all semaphore locks that belong to the specified process.
; This is used by termination.
; We do not test Ram semaphores.
;**********************************************************************
pid				EQU  WORD PTR [BP+6]
priority		EQU  BYTE PTR [BP-2*1]
TssId			EQU  WORD PTR [BP-2*2]
DsSave			EQU  WORD PTR [BP-2*3]
	ENTER	2*3,0
	MOV		DsSave,DS
	MOV		AX,DGroup
	MOV		DS,AX
	MOV		BX,oPcbRun					;BX has offset of our pcb
	MOV		AL,BYTE PTR pcbPriority[BX]
; Save our priority so we will know if it gets changed.
	MOV		priority,AL
	MOV		BX,pid						;BX has offset of pcb	
	MOV		DX,pcbTssId[BX]
	MOV		TssId,DX					;Save TssId
	PUSHF
	CLI									;DISABLE
	LES		BX,pSemNodeFirst
; ES:BX points to next semRec.
SpTestLast:
; Check for end of list of open semaphores.
	CMP		BX,OFFSET DGroup: pSemNodeFirst
	JE		SpClearFinished
	TEST	semFlags,semFlagRamSem
	JNZ		SpNextSemOffset				;It's a Ram semaphore
	TEST	semLockFlag,1				;Is semaphore locked?
	JZ		SpNextSemOffset				;No
	CMP		semOwnerTssId,DX			;Does it belong to specified tss?
SpTestOwner:
	JNE		SpNextSemOffset
; Set flag in this semaphore to indicate the owner was killed.
; The next process to lock the semaphore will get back ercSemOwnerKilled.
	OR		semFlags,semFlagOwnerKilled
; Increment the use count so that the record will not be deleted
; until we are finished with it.
	INC		semUseCount
	PUSH	ES							;Save pointer to semRec
	PUSH	BX
; Clear the semaphore and wake up any waiting processes
	PUSH	semHandle
	CALL	SemClear					
	POP		BX							;Restore pointer to semRec
	POP		ES
	PUSH	AX							;Save erc
; Decrement the use count for this Ram semaphore.
	DEC		semUseCount
	PUSH	sempNext					;Save pointer to next semaphore
; Delete the semaphore record if it is not being used.
	CALL	SemDeallocIfUnusedSub
; Set ES:BX to point to next semaphore.  
	POP		BX							
	POP		ES
	POP		AX							;Restore erc
	OR		AX,AX
	JNZ		SpClearError
	MOV		DX,TssId					;Restore DX to TssId
	JMP		SpTestLast
SpClearError:
; Error detected by SemClear
	POPF								;ENABLE
	JMP		SHORT SpRet

SpNextSemOffset:
; Load offset of next sem. 
	MOV		BX,WORD PTR sempNext
	JMP		SpTestLast

SpClearFinished:
; We have cleared all the semaphores that belong to the specified user.
; Check to see if our priority got changed.
	POPF								;ENABLE
	MOV		BX,oPcbRun					;BX has offset of our pcb
	MOV		AL,priority					;AL has our starting priority
	CMP		AL,BYTE PTR pcbPriority[BX]
	JE		SpRetOk
; Our priority was raised by a process that wanted the lock so 
; that we would hurry up and finish, or we raised our priority 
; so that we would not lose control when we sent a message
; to a waiting processs. Restore our priority.
	PUSH	AX
	CALL	ChangePriority
	JMP		SHORT SpRet
SpRetOk:
	XOR		AX,AX						;erc = ercOK
SpRet:
	MOV		DS,DsSave
	LEAVE
	RET		2
SemClearProcessLocks ENDP

SemDeallocIfUnused PROC FAR
;**********************************************************************
; SemDeallocIfUnused(pSemRec)
; Interrupts are disabled.
;**********************************************************************
pSemRec		EQU DWORD PTR [BP+6]
	ENTER	0,0
	LES		BX,pSemRec
	CALL	SemDeallocIfUnusedSub
	LEAVE
	RET		4
SemDeallocIfUnused ENDP

SemDeallocIfUnusedSub PROC NEAR
;**********************************************************************
; On entry, ES:BX points to semRec to deallocate if unused
; Interrupts are disabled.
; On return, 
;	ES:BX is no longer valid if semaphore record was deallocated.
;**********************************************************************
%IF(%CheckConsistency) THEN (
; Make sure semRec is in use.
	TEST	semFlags,semFlagInUse_mask
	JNZ		SdInUseOK
	%CallDebugger
SdInUseOK:
)FI
	XOR		EAX,EAX
	CMP		semOpenCount,AX
	JNE		SdRet					;openCount not zero
	CMP		semLockCount,AX
	JNE		SdRet					;lockCount not zero
	CMP		semUseCount,AL
	JNE		SdRet					;useCount not zero
; Check to see if there is a process waiting.
	LEA		DX,sempFirstWaiting
	CMP		DX,sempFirstWaiting
	JNE		SdRet					;process is waiting
; This semaphore can be deallocated.
; Zero the handle portion of the semaphore to invalidate it.
	MOV		semHandle,0
; Remove the sem from whatever queue it is on.
	%DeQueueSem
	%DeallocSem
	TEST	semFlags,semFlagRamSem
	JZ		SdCheckName
; This is a Ram semaphore, so we must dequeue it from the list of Ram
; semaphores that belong to this user.
	ADD		BX,sempNextUserOffset
	%DeQueueSem
; Note that we do not check sempsbName for Ram semaphore.  
	JMP		SHORT SdRet
SdCheckName:
	CMP		sempsbName,0
	JE		SdRet
; Deallocate the name record
	LES		BX,sempsbName
%IF(%CheckConsistency) THEN (
	OR		semFlags,semFlagInUse_mask
)FI
	%DeallocSem
SdRet:
	RET
SemDeallocIfUnusedSub ENDP

SemError PROC NEAR
;**********************************************************************
; SemError
; AX has erc on entry and return.
;**********************************************************************
	ENTER	0,0
	PUSH	DS						;Save DS
	PUSH	AX						;Parameter to CheckErcBreak
	MOV		AX,DGroup
	MOV		DS,AX
	CALL	CheckErcBreak			;Returns with erc in AX
	POP		DS						;Restore DS
	LEAVE
	RET		
SemError ENDP
SemLock PROC FAR
;**********************************************************************
; SemLock: PROCEDURE(sh, timeout) ErcType PUBLIC REENTRANT; 
;**********************************************************************
timeout		EQU  WORD PTR [BP+6]
sh			EQU DWORD PTR [BP+8]
pSemRec		EQU DWORD PTR [BP-2*2]
DsSave		EQU  WORD PTR [BP-2*3]
	ENTER	2*3,0
	%TestSh(SlRamSemaphore, SlsInvalidHandle, SlsCheckOptions)
SlsOptionsValid:
; ES:BX now points to valid system semRec
	%IncCounter (cSemLock)

SlsRetry:
; CX has DGroup
	STR  	DX						;DX = our TssId
	BTS		semLockFlag,0
	JC		SlsAlreadyLocked
; We have the system semaphore.
; Save owner TSS in high order word of semaphore.
	MOV		semOwnerTssId,DX
SlsTestOwnerDied:
; Check to see if the owner died
	TEST	semFlags,semFlagOwnerKilled
	JNZ		SlsOwnerKilled
	INC		semLockCount
	XOR		AX,AX					;erc = ercOK
	LEAVE
	RET		6

SlTimeout:
	%Error(ercSemTimeout)
	JMP		SlRestoreDsRet

SlsCheckOptions:
; Come here if the options in the semaphore record are anything
; other than zero.
	CALL	CheckSemOptions
	JZ		SlsOptionsValid
	JMP		SlRet


SlsAlreadyLocked:
; See if we already own the lock.
	CMP		DX,semOwnerTssId
	JE		SlsTestOwnerDied

; We will have to wait for the system semaphore
; Set the high byte of the semaphore to indicate a process is waiting
	%SetSemWaiting
	MOV		DsSave,DS
	MOV		DS,CX					;CX = DGroup
	MOV		CX,timeout				;CX = timeout parameter
	JCXZ	SlTimeout				;timeout if parameter is zero
	PUSHF
	CLI								;DISABLE
; We must test the lock once again with interrupts disabled.  
; It could have become clear since the time we first tested.
	TEST	semLockFlag,1
	JZ		SlsClearNow
; Alloc a wait record and queue it to the semRec.
	%IncCounter (cSemWaitForLock)
	CALL	SemAllocWaitSub
	JNZ		SlErcRet				;AX = ercNoSemMem
; On return, FS:SI points to waitRec.
; Check to see if the owner of the lock has a priority lower than ours.
; If the semaphore was set with SemSet, it will not have an owner.
	MOV		AX,semOwnerTssId
	OR		AX,AX
	JZ		SlsSuspend				;No owner
	SUB		AX,8
	MOV		GS,AX					;GS = data pointer to TSS of owner
	MOV		DI,oPcbRun				;DI points to our pcb
	MOV		DL,pcbPriority[DI]		;DL = our priority
	MOV		DI,GS:[tss_oPcb]		;DI points to owner pcb
	MOV		AL,pcbPriority[DI]		;AL = owner priority
	CMP		AL,DL					;owner priority, our priority
	JA		SlsChangePrio			;Owner priority is less
; Our priority (in DL) is numerically above or equal to priority of owner.
; This means our priority is equal to or lower than priority of owner.

SlsSuspend:
; FS:SI points to semWaitRec
	%IncCounter (cSemWaitForLock)
	CALL	SemWaitForClearSub
; WaitRec has been deleted.
	POPF							;ENABLE
; Check for timeout.  If the timeout field of the waitRec is zero,
; then the wait timed out.
	OR		AX,AX					;AX = timeout value
	JE		SlTimeout
	LES		BX,sh
	SUB		BX,semLockOffset		;ES:BX points to system semRec
	MOV		DS,DsSave				;Restore user's DS
	MOV		CX,DGroup
	JMP		SlsRetry	
	
SlsChangePrio:
; The owner of the lock has a priority numerically greater than ours.
; This means the owner has priority less than ours.
; We must raise his priority to be equal to ours.
; DI points to pcb of owner of lock.
; DL = our priority
; AL = owner priority
	CMP		semPriority,0
	JNE		SlsSavedPrio
; If semRec.semPriority <> 0, then save the priority of the owner in
; the semaphore record.  This is so that we will be able to change
; the owner priority back to what it was when he releases the lock.
	MOV		semPriority,AL
SlsSavedPrio:
	PUSH	FS						;Save pointer to waitRec
	PUSH	SI
	PUSH	DX						;DL has our priority
	PUSH	DI						;DI has oPcbOwner
	%IncCounter (cSemChangePriority)
	CALL	ChangeProcessPriority	;(oPcbOwner, pcb.priority)
	POP		SI						;Restore pointer to waitRec
	POP		FS
	JMP		SlsSuspend

SlsClearNow:
	POPF							;ENABLE
	MOV		DS,DsSave
	MOV		CX,DGroup
	JMP		SlsRetry

SlRamSemaphore:
;**********************************************************************
; If the sg of the semaphore handle does not match sgSemNode, then
; we assume it's a Ram semaphore.
;**********************************************************************
	%IncCounter (cSemLockRam)
SlrRetry:
	BTS		WORD PTR ES:[BX],0		;Test and Set Ram semaphore
	JC		SlrAlreadyLocked		;Already locked
; We got the Ram semaphore.  The semaphore lock has been set.
	STR		WORD PTR ES:[BX+2]		;Store TssId in high order word
	XOR		AX,AX					;erc = ercOK
	LEAVE
	RET		6

SlrTimeout:
	%Error(ercSemTimeout)
	JMP		SlRestoreDsRet

SlrAlreadyLocked:
	MOV		DsSave,DS
	MOV		DS,CX					;CX = DGroup
	MOV		CX,timeout				;CX = timeout parameter
SlrCheckTimeout:
	JCXZ	SlrTimeout				;timeout if parameter is zero
	PUSHF
	CLI								;DISABLE
; We must test the lock once again with interrupts disabled.  
; It could have become clear since the time we first tested.
	BTS		WORD PTR ES:[BX],0		;Test and Set Ram semaphore
	JNC		SlrClearNow	
; Set the high byte of the Ram semaphore to indicate a process
; is waiting for it.
	%SetRamSemWaiting
; Alloc a wait record and queue it to the semRec.
	%IncCounter (cSemWaitForLockRam)
	PUSH	sh						
	CALL	AllocRamSemRec
	JNZ		SlErcRet				;AX = ercNoSemMem
; ES:BX points to Ram semRec.
	MOV		WORD PTR pSemRec,BX
	MOV		WORD PTR pSemRec+2,ES
; Alloc a wait record and queue it to the Ram semRec.
	MOV		CX,timeout				;CX = timeout parameter
	CALL	SemAllocWaitSub
	JNZ		SlErcRet				;AX = ercNoSemMem
	CALL	SemWaitForClearSub
; WaitRec has been deleted by SemWaitForClearSub.
	PUSH	AX						;Save timeout value
	LES		BX,pSemRec
; Deallocate the semRec we created for the Ram semaphore.
	CALL	SemDeallocIfUnusedSub
	POP		CX						;CX = 0 if semaphore timed out.
	POPF							;ENABLE
; Check for timeout.  If the timeout field of the waitRec is zero,
; then the wait timed out.
	LES		BX,sh
	JMP		SlrCheckTimeout

SlsOwnerKilled:
; We were the first to lock this system semaphore after the owner died.
; Turn off the owner died flag and return ercSemOwnerKilled.
	MOV		semLockCount,1
	AND		semFlags,0FFh-semFlagOwnerKilled
	%Error(ercSemOwnerKilled)
	JMP		SHORT SlRet

SlsInvalidHandle:
	%Error(ercSemInvalidHandle)
	JMP		SHORT SlRet

SlrClearNow:
	XOR		AX,AX					;erc = ercOK
	STR		WORD PTR ES:[BX+2]		;Store TssId in high order word

SlErcRet:
; AX has erc
	POPF							;ENABLE
SlRestoreDsRet:
	MOV		DS,DsSave
SlRet:
	LEAVE
	RET		6						;AX has erc
SemLock ENDP

SemMuxWait PROC FAR
;**********************************************************************
;SemMuxWait: PROCEDURE(pIndexRet, pSemaphoreList, timeOut) ErcType 
;**********************************************************************
pIndexRet		EQU DWORD PTR [BP+12]
pSemaphoreList	EQU DWORD PTR [BP+8]
timeout			EQU  WORD PTR [BP+6]
pShFirst		EQU DWORD PTR [BP-2*2]
pShNext			EQU DWORD PTR [BP-2*4]
sh				EQU DWORD PTR [BP-2*6]
pMsg			EQU DWORD PTR [BP-2*8]
DsSave			EQU  WORD PTR [BP-2*9]
semCount		EQU  WORD PTR [BP-2*10]
index			EQU  WORD PTR [BP-2*11]
erc				EQU  WORD PTR [BP-2*12]
exch			EQU  WORD PTR [BP-2*13]
waitTimeout		EQU  WORD PTR [BP-2*14]
fClearSemFound	EQU  BYTE PTR [BP-2*15]
	ENTER	2*15,0
	MOV		DsSave,DS
	MOV		AX,DGroup
	MOV		DS,AX
	%IncCounter (cSemMuxWaitCalled)
	MOV		erc,0					;Assume ercOK
	MOV		index,0
; Get pointer to list of semaphores to wait for.
	LES		BX,pSemaphoreList
	MOV		AX,muxSemCount
	OR		AX,AX
	JZ		MwRetIndex				;No semaphores in list
	MOV		semCount,AX				;Save count
	LEA		BX,muxSemSh
	MOV		WORD PTR pShFirst,BX	;Save pointer to first semaphore
	MOV		WORD PTR pShFirst+2,ES
	MOV		WORD PTR pShNext,BX
	MOV		WORD PTR pShNext+2,ES
; Start critical section now -- we don't want a semaphore to be cleared
; while we are in the middle of things.
	PUSHF
; DISABLE
	CLI								
	JMP		SHORT MwTestNext1
; Make a first pass through the list to see if any semaphore is already clear.
MwTestNext:
	LES		BX,pShNext
MwTestNext1:
	MOV		EAX,DWORD PTR ES:[BX]	;Get next handle
	MOV		sh,EAX
	%TestSh(MwRamSemaphore, MwInvalidHandle, MwCheckOptions)
MwssOptionsValid:
; ES:BX now points to system semRec
	TEST	semLockFlag,1
	JNZ		MwSemNotClear	
; Check to see if the owner died.
	TEST	semFlags,semFlagOwnerKilled
	JNZ		MwOwnerKilled
MwSemClear:
; A semaphore is clear -- no waiting.
	POPF							;ENABLE
	JMP		MwRetIndex

MwCheckOptions:
; Come here if the options in the semaphore record are anything
; other than zero.
	CALL	CheckSemOptions
	JZ		MwssOptionsValid
	POPF							;ENABLE
	JMP		MwRet

MwRamSemaphore:
	TEST	BYTE PTR ES:[BX],1
	JZ		MwSemClear

MwSemNotClear:
; This semaphore is not clear -- see if there is another.
	INC		index
	MOV		AX,index
	CMP		AX,semCount
	JE		MwNoneClear				;All semaphores are set
	ADD		WORD PTR pShNext,muxSemListSize
	JMP		MwTestNext

MwNoneClear:
; If no semaphore in the list is clear, we must wait for one to become clear.
; Interrupts are disabled.
	MOV		AX,timeout				;AX = timeout parameter
	OR		AX,AX
	JZ		MwTimeout				;timeout if parameter is zero
	MOV		waitTimeout,AX			;save value to put in waitRec
; Make a second pass through the list to allocate a wait
; record for each semaphore.
; Make sure we have enough waitRecs for this muxWait.
	MOV		AX,semCount
	CMP		AX,nSemNodeFree
	JA		MwNoWaitRec
	MOV		EAX,pShFirst
	MOV		pShNext,EAX
	MOV		index,0
MwAllocWait:
	LES		BX,pShNext
	MOV		EAX,DWORD PTR ES:[BX]	;Get next handle
	MOV		sh,EAX
	MOV		BX,sh+2
	CMP		BX,sgSemNode			;Is this a system semaphore handle?
	MOV		ES,BX					;ES = sg of handle
	MOV		BX,AX					;ES:BX = sh
	JNE		MwRamWaitRec			;Must be a Ram semaphore
	SUB		BX,semLockOffset		
; ES:BX points to semRec
; Set the high byte of the semaphore to indicate a process
; is waiting for it.
	%SetSemWaiting
; Alloc a wait record and queue it to the semRec.
; Use timeout value that will be either original parameter or 0FFFF
	MOV		CX,waitTimeout
	CALL	SemAllocWaitSub
; FS:SI points to semWaitRec
	JNZ		MwNoWaitRec

MwSetMux:
; Remember that this waitRec is for a MuxWait.  When we wake up
; a waiting process, we must send a message rather than unsuspend.
	OR		semWaitFlags,semWaitFlagMux_mask
	INC		index
	MOV		AX,index
	CMP		AX,semCount
	JE		MwNoneClearRecsAllocated
; Set waitTimeout to -1 after the first one so that if we get a
; timeout, we won't get a message for every semaphore.
	MOV		waitTimeout,0FFFFh
	ADD		WORD PTR pShNext,muxSemListSize
	JMP		MwAllocWait

MwRamWaitRec:
; Set the high byte of the semaphore to indicate a process
; is waiting for it.
	%SetRamSemWaiting
	PUSH	sh						
	CALL	AllocRamSemRec
	JNZ		MwNoWaitRec				;AX = ercNoSemMem
; ES:BX points to Ram semRec.
; Increment the use count so that the record will not be deleted
; until we are finished with it.
	INC		semUseCount
	MOV		CX,waitTimeout
	CALL	SemAllocWaitSub
; FS:SI points to semWaitRec
	JNZ		MwNoWaitRec
	JMP		MwSetMux

MwNoneClearRecsAllocated:
; We have allocated a waitRec for each semaphore in the list.
; FS:SI points to last waitRec allocated
	POPF							;ENABLE
	%IncCounter (cSemMuxWait)
	MOV		AX,semWaitExch
	MOV		exch,AX					;Save exch for Check
	PUSH	AX						;exch
	LEA		AX,pMsg					;We don't care about the message
	PUSH	SS							
	PUSH	AX
	CALL	WAIT
; Continue when a message is sent to our exchange.
; Either the wait timed out, or a semaphore was cleared.
	PUSHF	
	CLI								;DISABLE

MwCheck:
; We could have more than one message, so Check to clear any other
; messages.
	PUSH	exch
	LEA		AX,pMsg					;We don't care about the message,
	PUSH	SS							
	PUSH	AX
	CALL	Check
	OR		AX,AX
	JZ		MwCheck
; Go through the list bottom to top to remove waiting record for 
; each semaphore.  We only remove the waitRec that we created.
; We know we created it for MuxWait because our TssId matches the
; TssId in the waitRec.  When we are through, we have the index of 
; the first semaphore in the list that is clear.
; Note that index = count and pShNext points to last sh
; from loop above where we allocated waitRecs.
	DEC		index						;index = last index
	MOV		fClearSemFound,0
MwFindClearSem:
	LES		BX,pShNext
	MOV		EAX,DWORD PTR ES:[BX]	;Get next handle
	MOV		sh,EAX
	MOV		BX,sh+2
	CMP		BX,sgSemNode			;Is this a system semaphore handle?
	MOV		ES,BX					;ES = sg of handle
	MOV		BX,AX					;ES:BX = sh
	JNE		MwTestClearRamSem		;Must be a Ram semaphore
	SUB		BX,semLockOffset		
; ES:BX points to next system semRec
; Check the next semaphore to see if it is clear.
	TEST	semLockFlag,1
	JNZ		MwCheckWaitRecs			;Not clear
; System semaphore is clear.  Check owner died.
	TEST	semFlags,semFlagOwnerKilled
	JZ		MwSaveIndexClear		;Owner did not die
	PUSH	ES						;Save sh
	PUSH	BX
	%Error(ercSemOwnerKilled)
	POP		BX						;Restore sh
	POP		ES
	MOV		erc,AX
MwSaveIndexClear:
; Return the index of the semaphore.
	LFS		SI,pIndexRet
	MOV		CX,index
	MOV		FS:[SI],CX
	MOV		fClearSemFound,0FFh
MwCheckWaitRecs:
; ES:BX points to next semaphore in list.  
	LEA		AX,sempFirstWaiting
; AX points to head of list of waitRecs
; See if there are any waitRecs for this semaphore.
	CMP		AX,WORD PTR sempFirstWaiting
	JE		MwCheckLast				;no waitRecs
; This semaphore has a waitRec
	LFS		SI,sempFirstWaiting
	STR		DX						;DX has our TSS id
MwTestWaitRecId:
; Check to see if this waitRec belongs to the process that is running
	CMP		DX,semWaitTssId
	JNE		MwPointNextWaitRec		;Not our process
; This waitRec belongs to our process.  See if it timed out.
	MOV		CX,semWaitTimeout
	OR		CX,CX
	JNZ		MwRemoveWaitRec			;did not time out
; This waitRec timed out.  Set erc unless we have already set it.
	CMP		erc,0
	JNE		MwRemoveWaitRec
	PUSH	ES						;Save sh
	PUSH	BX
	PUSH	FS						;Save pSemWaitRec
	PUSH	SI
	%Error(ercSemTimeout)
	POP		SI						;Restore pSemWaitRec
	POP		FS
	POP		BX						;Restore sh
	POP		ES
	MOV		erc,AX
MwRemoveWaitRec:
	%DeQueueSemWaitRec
	%DeallocSemWaitRec
; Note that we do not try to find another waitRec for this semaphore.
	CALL	SemDeallocIfUnusedSub
	JMP		SHORT MwCheckLast

MwTestClearRamSem:
; ES:BX points to next Ram semaphore.  We must find the semNode record
; that corresponds to this Ram semaphore.
	MOV		EAX,sh
	CALL	FindRamSemRec
	JNZ		MwRamSemRecNotFound
; ES:BX points to semNode for Ram semaphore.
; Decrement the use count for this Ram semaphore.
	DEC		semUseCount
%IF(%CheckConsistency) THEN (
	JNS		MwUseCountOk
	%CallDebugger
MwUseCountOk:
)FI
	LFS		SI,sh					;FS:SI points to Ram semaphore
	TEST	BYTE PTR FS:[SI],1		;Is Ram semaphore clear?
	JNZ		MwCheckWaitRecs			;Not clear
	JMP		MwSaveIndexClear		;Save index of clear semaphore

MwRamSemRecNotFound:
; This is an internal error if we cannot find the semRec for
; the Ram semaphore.
	%IncCounter (cRamMuxSemRecNotFound)
	%CallDebugger
	JMP		SHORT MwInvalidHandle

MwPointNextWaitRec:
	MOV		SI,WORD PTR semWaitpNext
MwTestLastWaitRec:
; Check to see if there is another waitRec for this semaphore.
; AX has offset of sem.sempFirstWaiting (head of wait list).
	CMP		AX,SI
	JNE		MwTestWaitRecId

MwCheckLast:
; See if we have checked all the semaphores.
	SUB		WORD PTR pShNext,muxSemListSize
	DEC		index
	JNS		MwFindClearSem
; We have gone through the list of semaphores.
	POPF							;ENABLE
	CMP		erc,0
	JNE		MwRetErc
; Check to see if we found a clear semaphore.  
; If not, go wait again.
	CMP		fClearSemFound,0
	JNE		MwRetErc
	PUSHF
	CLI								;DISABLE
	JMP		MwNoneClear
MwTimeout:
	POPF							;ENABLE
	%Error(ercSemTimeout)
	JMP		SHORT MwRet

MwInvalidHandle:
	POPF							;ENABLE
	%Error(ercSemInvalidHandle)
	JMP		SHORT MwRetIndexErcAX
MwOwnerKilled:
	POPF							;ENABLE
	%Error(ercSemOwnerKilled)
MwRetIndexErcAX:
	MOV		erc,AX
	JMP		SHORT MwRetIndex
MwNoWaitRec:
	POPF							;ENABLE
	%IncCounter(cNoSemMem)
	%Error(ercNoSemMem)
	JMP		SHORT MwRet
MwRetIndex:
; Enabled
; Return the index of the semaphore.
	LES		BX,pIndexRet
	MOV		AX,index
	MOV		ES:[BX],AX
MwRetErc:
	MOV		AX,erc
MwRet:
	MOV		DS,DsSave
	LEAVE
	RET		10
SemMuxWait ENDP

SemNotify PROC FAR
;**********************************************************************
; SemNotify: PROCEDURE(sh, timeout, exch) ErcType PUBLIC REENTRANT; 
;**********************************************************************
sh			EQU DWORD PTR [BP+10]
timeout		EQU  WORD PTR [BP+8]
exch		EQU  WORD PTR [BP+6]
DsSave		EQU  WORD PTR [BP-2*1]
	ENTER	2*1,0
	MOV		DsSave,DS
	MOV		AX,DGroup
	MOV		DS,AX
	%IncCounter (cSemNotify)
	%TestSh(SnInvalidHandle, SnInvalidHandle, SnCheckOptions)
SnOptionsValid:
; ES:BX now points to system semRec
	TEST	semLockFlag,1
	JZ		SnSend					;not locked, so send a message
; The semaphore is locked.  See if there is a timeout.
; If no timeout, send a message now.  Application will get timeout.
	MOV		CX,timeout
	JCXZ	SnSend					;No timeout	
; Alloc a wait record and queue it to the semRec.
	PUSHF
	CLI								;DISABLE
	CALL	SemAllocWaitSub			;CX = timeout value
	JNZ		SnAllocError
; Set the waitRec with the exchange specified by the caller.
	MOV		AX,exch
	MOV		semWaitExch,AX
; Set the notifyFlag in the waitRec so we will know the waiter is
; for SemNotify.
	OR		semWaitFlags,semWaitFlagNotify_mask
	POPF							;ENABLE
	XOR		AX,AX					;erc = ercOK
	JMP		SHORT SnRet

SnCheckOptions:
; Come here if the options in the semaphore record are anything
; other than zero.
	CALL	CheckSemOptions
	JZ		SnOptionsValid
	LEAVE
	RET		8

SnInvalidHandle:
	%Error(ercSemInvalidHandle)
	LEAVE
	RET		8

SnAllocError:
	POPF							;ENABLE
	JMP		SHORT SnRet

SnSend:
	PUSH	exch
	PUSH	semHandle
	CALL	SEND
SnRet:
	MOV		DS,DsSave
	LEAVE
	RET		8
SemNotify ENDP

SemQuery PROC FAR
;**********************************************************************
; SemQuery: PROCEDURE(sh, pbSemRecordRet, cbSemRecordRet, 
;	pbSemNameRet, cbSemNameRet) ErcType PUBLIC REENTRANT
;**********************************************************************
sh				EQU DWORD PTR [BP+18]
pbSemRecordRet	EQU DWORD PTR [BP+14]
cbSemRecordRet	EQU  WORD PTR [BP+12]
pbSemNameRet	EQU DWORD PTR [BP+8]
cbSemNameRet	EQU  WORD PTR [BP+6]
pSemRec			EQU DWORD PTR [BP-2*2]
DsSave			EQU  WORD PTR [BP-2*3]
	ENTER	2*3,0
	MOV		DsSave,DS
	MOV		AX,DGroup
	MOV		DS,AX
	%IncCounter (cSemQuery)
	PUSHF
	CLD
	CLI								;DISABLE
; Note that we would like to use the TestSh macro here, but we
; can't quite do it because sh could be an LDT pointer.
	MOV		BX,sh+2
	CMP		BX,sgSemNode			;Is this a system semaphore handle?
	JNE		SqRamSem				;Must be a Ram semaphore
	MOV		ES,BX					;ES = sg of handle
	MOV		BX,sh					;ES:BX = sh
; Verify that the semHandle in the semaphore record is the same as
; the handle passed in.
	CMP		BX,semHandle-semLockOffset
	JNE		SqInvalidHandle
	SUB		BX,semLockOffset		;ES:BX points to system sem rec
; We have a valid system semaphore handle in ES:BX
SqReturnSemRec:
; ES:BX now points to system semRec.  Return semRec to caller.
	MOV		WORD PTR pSemRec+2, ES	;Save pointer to semRec
	MOV		WORD PTR pSemRec,BX		
	LEA		SI,semLockFlag
	MOV		DS,pSemRec+2			;DS:SI points to semLock
	LES		DI,pbSemRecordRet		;ES:DI points to semRecordRet
	MOV		CX,cbSemRecordRet
	REPNZ	MOVSB
; Assume the semaphore has no name -- set zero in sbSemNameRet.
	LES		BX,pbSemNameRet
	MOV		BYTE PTR ES:[BX],0
; Check to see if the semaphore has a name.
	LES		BX,pSemRec				;Restore pointer to semRec
	CMP		semPsbName,0			;Is name pointer null?
	JE		SqRetOK					;Yes, so no name
	LDS		SI,semPsbName			;DS:SI points to sbName
	MOV		CX,cbSemNameRet
	LES		DI,pbSemNameRet
	REPNZ	MOVSB
	JMP		SqRetOk

SqRamSem:	
;**********************************************************************
; If the sg of the semaphore handle does not match sgSemNode, then
; we assume it's a Ram semaphore.
;**********************************************************************
	%IncCounter (cSemQueryRam)
	MOV		EAX,sh
	CALL	FindRamSemRec
	JNZ		SqInvalidHandle			;Ram semRec not found
; ES:BX points to semRec of ram semaphore.
; The piece we are missing is the lock bit.  
; In order to access the Ram semaphore, we must set up our LDT pointer
; to be the same as the process waiting for the semaphore because
; the Ram semaphore could be an LDT semaphore.
	LEA		AX,sempFirstWaiting
	CMP		AX,WORD PTR sempFirstWaiting
	JE		SqNoneWaiting
; We have a process waiting, so get the TssId of the process waiting
; for the lock.
	LFS		SI,sempFirstWaiting
	MOV		AX,semWaitTssId
	SUB		AX,8
	MOV		GS,AX					;GS = data pointer to TSS of waiter
	SLDT	DX						;Save our LDT
	LLDT	GS:[rTss386LDT]			;Load LDT of waiter
; Now our LDT is the same as the process waiting for the semaphore,
; so we can load the pointer to the semaphore.
	LGS		DI,sh					;GS:DI points to Ram semaphore
	MOV		EAX,GS:[DI]
	MOV		semLockAndOwner,EAX		;Store lock and owner in semRec
	XOR		AX,AX					;Make GS zero so that caller won't 
	MOV		GS,AX					;get a GP fault later
	LLDT	DX						;Restore our LDT
%IF(%Debug) THEN (
	TEST	WORD PTR sh+2,maskLdt
	JZ		SqGdt
	%IncCounter(cSemQueryRamLDT)
SqGdt:
)FI
; Now that we have set up the lock and owner, we can move
; the semRec into the caller's space.
	JMP		SqReturnSemRec

SqNoneWaiting:	
; If the semRec does not have any process waiting, assume it is clear.
	MOV		semLockAndOwner,0
	JMP		SqReturnSemRec

SqInvalidHandle:
	%Error(ercSemInvalidHandle)
	JMP		SHORT SqRet
SqRetOK:
	XOR		AX,AX					;Erc = ercOK
SqRet:
	POPF							;ENABLE
	MOV		DS,DsSave				;Restore DS
	LEAVE
	RET		16
SemQuery ENDP

;**********************************************************************
;	SemRestoreDS: PROCEDURE(DS)
;	This procedure Restores the passed DS 
;**********************************************************************
SemRestoreDS	PROC	FAR
	ENTER 0,0
	MOV		DS,[BP+6]
	LEAVE
	RET  2
SemRestoreDS	ENDP

;**********************************************************************
;	SemSaveDS: PROCEDURE WORD
;	This procedure returns DS in AX, and sets DS to DGroup 
;**********************************************************************
SemSaveDS	PROC	FAR
	MOV		AX,DS
	MOV		BX,DGroup
	MOV		DS,BX
	RET
SemSaveDS	ENDP

SemSet PROC FAR
;**********************************************************************
; SemSet: PROCEDURE(sh) ErcType PUBLIC REENTRANT; 
;**********************************************************************
	MOV		DI,SP
sh	EQU	DWORD PTR SS:[DI+4]
	%TestSh(SsRamSemaphore, SsInvalidHandle, SsCheckOptions)
SsOptionsValid:
; ES:BX now points to system semRec
	%IncCounter (cSemSet)
	MOV		semLockAndOwner,1		;Don't store Tss in high word
	XOR		AX,AX
	RET		4

SsCheckOptions:
; Come here if the options in the semaphore record are anything
; other than zero.
	CALL	CheckSemOptions
	JZ		SsOptionsValid
	RET		4

SsInvalidHandle:
	%Error(ercSemInvalidHandle)
	RET		4

SsRamSemaphore:
; If the sg of the semaphore handle does not match sgSemNode, then
; we assume it's a Ram semaphore.
	%IncCounter (cSemSetRam)
	MOV		DWORD PTR ES:[BX],1		;Don't store Tss in high word
	XOR		AX,AX
	RET		4
SemSet ENDP

%IF(0) THEN (
; No longer used, but save the code for possible future use.
SemUnChain PROC FAR
;**********************************************************************
; SemUnchain PROCEDURE:(pNode)
; Unchain entry from whatever list it is in.
;**********************************************************************
	ENTER 0,0
	PUSH DS
	LES  BX,DWORD PTR[BP+6]		;ES:BX points to node to unchain
	LDS  SI,ES:pNext[BX]		;DS:SI points to next entry or list head
	MOV  DI,ES:raPrev[BX]		;AX:DI points to prev entry or list head 
	MOV  AX,ES:saPrev[BX]
; Unchain this node from the prev pointer of the next entry.
	MOV  raPrev[SI],DI
	MOV  saPrev[SI],AX
; Unchain this node from the next pointer of the previous entry.
	MOV  ES,AX
	MOV  ES:raNext[DI],SI
	MOV  ES:saNext[DI],DS
	POP  DS
	LEAVE
	RET  4
SemUnChain ENDP
)FI

SemTimerInt PROC FAR
;**********************************************************************
;	Called whenever the timer goes off
;	Check to see if any semaphores have timed out.
;**********************************************************************
pSemWaitNext		EQU DWORD PTR [BP-2*2]
fTimeoutPending		EQU  BYTE PTR [BP-2*3]
	ENTER	2*3,0
	MOV		fTimeoutPending,0
	PUSHF
	CLI									;DISABLE
StRestart:
	LES		BX,pSemNodeFirst
; ES:BX points to next semRec.
StSetup:
	MOV		DX,semTrbCounterReload
; DX has value to subtract from each semWaitRec.timeout.

StTestLast:
; Check for end of list of open semaphores.
	CMP		BX,OFFSET DGroup: pSemNodeFirst
	JE		StFinishedWaitRecs
StCheckWaiting:
; Check to see if semaphore has any processes waiting
	LEA		AX,sempFirstWaiting
	CMP		AX,WORD PTR sempFirstWaiting
	JNE		StHasWaitRec
StNextSemOffset:
; Load offset of next semaphore. 
	MOV		BX,WORD PTR sempNext
	JMP		StTestLast

StHasWaitRec:
; AX has offset of sem.sempFirstWaiting (head of wait list).
; DX has semTrbCounterReload
; We don't want to do this whole thing with interrupts disabled.
; Increment the use count so that the record will not be deleted
; until we are finished with it.
	INC		semUseCount
	POPF
	PUSHF
	CLI
; How can we be sure that the current semaphore has not been dequeued?
; We can test the next pointer and see if it has changed.
; Decrement the use count for Ram semaphore.
	DEC		semUseCount
	LFS		SI,sempFirstWaiting

StTestTimeout:
%IF(%TestClearSem) THEN (
; If :SuppressRamSemaphoreHack: yes, then don't check for 
; clear Ram semaphore.
	CMP		vfRamSemaphoreHack,0
	JE		EndSemClearTest
; Check to see if we are waiting for a semaphore that is clear.
; This would be an error condition.
; Check the handle to see if it's a GDT handle.
; If not, then we can't access it.
	TEST	WORD PTR semHandle+2,4
	JNZ		EndSemClearTest					;LDT handle
	LGS		DI,semHandle
	TEST	WORD PTR GS:[DI],1				;Is Semaphore set
	JNZ		EndSemClearTest
; Semaphore is clear
; Check to see if we have already woken up the waiting process.
	TEST	semWaitFlags,semWaitFlagWokeUp_mask
	JNZ		EndSemClearTest
	INC		cWaitingForClearSem
	JMP		StWakeup
EndSemClearTest:
)FI

	MOV		CX,semWaitTimeout
	JCXZ	StPointNextWaitRec				;no timeout
	CMP		CX,0FFFFh
	JNE		StHasTimeout
; This waitRec does not have a timeout.

StPointNextWaitRec:
	MOV		SI,WORD PTR semWaitpNext
StTestLastWaitRec:
; Check to see if there is another waitRec for this sem.
; AX has offset of sem.sempFirstWaiting (head of wait list).
	CMP		AX,SI
	JE		StNextSemOffset					;end of waitRec queue
	JMP		StTestTimeout

StHasTimeout:
; CX has the waitRec timeout value.
; DX has semTrbCounterReload
	SUB		CX,DX
	JLE		StTimedOut
; Update the time left in the timeout for this waitRec.
	MOV		semWaitTimeout,CX
; Remember that we still have a timeout pending.
	MOV		fTimeoutPending,0FFh
	JMP		StPointNextWaitRec

StTimedOut:
; This waitRec has timed out.  Wake up the waiter(s).
	MOV		semWaitTimeout,0	
StWakeup:
	MOV		EAX,semWaitpNext			;Save pointer to next waitRec
	MOV		pSemWaitNext,EAX
; Check to see if this timeout was for a SemNotify.
	TEST	semWaitFlags,semWaitFlagNotify_mask
	JNZ		StNotify
	CALL	SemWakeupSub				;ES:BX, FS:SI preserved
; Note that the WaitRec may have been deleted if another process ran.
; The only way we can know is to check to see if the next pointer
; has changed.
	MOV		EAX,semWaitpNext
	CMP		EAX,pSemWaitNext
	JNE		StRestartWaitRecs
	MOV		DX,semTrbCounterReload		;Restore counterReload
	LEA		AX,sempFirstWaiting			;Restore offset of list head
	JMP		StPointNextWaitRec

StRestartWaitRecs:
; The waitRec was deallocated, so
; we must start at the beginning of the list again.
	%IncCounter(cRestartWaitRecs)
	JMP		StCheckWaiting

StNotify:
	PUSH	semWaitpNext				;Save pointer to next waitRec
	CALL	SemWakeupSub				;ES:BX points to semWaitRec
; SemWakeupSub Deleted the waitRec for this notify.
	POP		SI
	POP		FS							;Point to next waitRec
	MOV		DX,semTrbCounterReload		;Restore counterReload
	LEA		AX,sempFirstWaiting			;Restore offset of list head
; Check to see if there is another waitRec
	CMP		AX,sempFirstWaiting
	JNE		StTestLastWaitRec			;process is waiting
; Check to see if this sem should be deleted.
	CMP		semOpenCount,0
	JNE		StTestLastWaitRec			;openCount not zero
	CMP		semLockCount,0
	JNE		StTestLastWaitRec			;lockCount not zero
	CMP		semUseCount,0
	JNE		StTestLastWaitRec			;useCount not zero
; This semaphore can be deallocated.
	PUSH	sempNext					;Save pointer to next sem
	CALL	SemDeallocIfUnusedSub
	POP		BX
	POP		ES							;Point to next sem
	JMP		StSetup

StFinishedWaitRecs:
	XOR		AX,AX
	CMP		fTimeoutPending,AL
	JE		StNoMoreTimeouts
; Zero cEvents to start the timer again
	MOV		semTrbcEvents,AX
	JMP		SHORT StRet
StNoMoreTimeouts:
; Set the counter and counterReload to 0FFFF so that the
; next time we set a timeout, we will start the timer and
; set the counter to the value of the timeout.
	DEC		AX							;AX = 0FFFFH
	MOV		semTrbCounter,AX
	MOV		semTrbCounterReload,AX
%IF(%TestClearSem) THEN (
; If testing for clear semaphores, keep the timer running so that we will
; keep checking for clear semaphores.
	MOV		semTrbCounter,10
	MOV		semTrbCounterReload,10		;1 second
	MOV		semTrbcEvents,0
)FI
StRet:
	POPF								;ENABLE
	LEAVE
	RET
SemTimerInt ENDP

SemWait PROC FAR
;**********************************************************************
; SemWait: PROCEDURE(sh, timeout) ErcType PUBLIC REENTRANT; 
;**********************************************************************
timeout		EQU  WORD PTR [BP+6]
sh			EQU DWORD PTR [BP+8]
pSemRec		EQU DWORD PTR [BP-2*2]
raSemRec	EQU  WORD PTR pSemRec
sgSemRec	EQU  WORD PTR pSemRec+2
DsSave		EQU  WORD PTR [BP-2*3]
	ENTER	2*3,0
	%TestSh(SwRamSemaphore, SwsInvalidHandle, SwsCheckOptions)
SwsOptionsValid:
; ES:BX now points to a valid system semRec
	%IncCounter (cSemWaitCalled)
	TEST	semLockFlag,1
	JNZ		SwsSemSet
; The system semaphore is not locked

SwsCheckOwnerDied:
	TEST	semFlags,semFlagOwnerKilled
	JNZ		SwsOwnerKilled
	XOR		AX,AX					;erc = ercOK
	LEAVE
	RET		6

SwsCheckOptions:
; Come here if the options in the semaphore record are anything
; other than zero.
	CALL	CheckSemOptions
	JZ		SwsOptionsValid
	JMP		SwRet

SwTimeout:
	%Error(ercSemTimeout)
	JMP		SwRestoreDsRet

SwsSemSet:
	MOV		DsSave,DS
	MOV		DS,CX					;CX = DGroup
	MOV		CX,timeout				;CX = timeout parameter
	JCXZ	SwTimeout				;timeout if parameter is zero
	PUSHF
	CLI								;DISABLE
; We must test the lock once again with interrupts disabled.  
; It could have become clear since the time we first tested.
	TEST	semLockFlag,1
	JZ		SwsClearNow
; Alloc a wait record and queue it to the semRec.
	CALL	SemAllocWaitSub
	JNZ		SwErcRet				;AX = ercNoSemMem
	%IncCounter (cSemWait)
	CALL	SemWaitForClearSub
; WaitRec has been deleted.
	POPF							;ENABLE
SwCheckTimeout:
; Check for timeout.  If the timeout field of the waitRec is zero,
; then the wait timed out.
	OR		AX,AX
	JE		SwTimeout
	XOR		AX,AX					;Erc = ercOK
	JMP		SwRestoreDsRet

SwsClearNow:
	POPF							;ENABLE
	MOV		DS,DsSave
	JMP		SwsCheckOwnerDied

SwRamSemaphore:
;**********************************************************************
; If the sg of the semaphore handle does not match sgSemNode, 
; then we assume it's a Ram semaphore.
;**********************************************************************
	%IncCounter (cSemWaitRamCalled)
	TEST	BYTE PTR ES:[BX],1
	JNZ		SwrSemSet
; The Ram semaphore is clear.  Easy out.
	XOR		AX,AX					;erc = ercOK
	LEAVE
	RET		6

SwrSemSet:
	MOV		DsSave,DS
	MOV		DS,CX					;CX = DGroup
	MOV		CX,timeout				;CX = timeout parameter
	JCXZ	SwTimeout				;timeout if parameter is zero
	PUSHF							;Save flags
	CLI								;DISABLE
; We must test the lock once again with interrupts disabled.  
; It could have become clear since the time we first tested.
	TEST	BYTE PTR ES:[BX],1
	JZ		SwrClearNow
	PUSH	sh						
	CALL	AllocRamSemRec
	JNZ		SwErcRet				;AX = ercNoSemMem
; ES:BX points to Ram semRec.
	MOV		WORD PTR pSemRec,BX
	MOV		WORD PTR pSemRec+2,ES
; Alloc a wait record and queue it to the Ram semRec.
	MOV		CX,timeout				;CX = timeout parameter
	CALL	SemAllocWaitSub
	JNZ		SwErcRet				;AX = ercNoSemMem
	%IncCounter (cSemWaitRam)
; Set the high byte of the Ram semaphore to indicate a process
; is waiting for it.
	LES		BX,sh
	OR		WORD PTR ES:[BX],100H
	CALL	SemWaitForClearSub
; WaitRec has been deleted.
	PUSH	AX						;Save erc
; Deallocate the semRec we created for the Ram semaphore.
	LES		BX,pSemRec
	CALL	SemDeallocIfUnusedSub
	POP		AX						;AX = 0 if semaphore timed out.
	POPF							;ENABLE
; Check for timeout.  If the timeout field of the waitRec is zero,
; then the wait timed out.
	JMP		SwCheckTimeout

SwsOwnerKilled:
	AND		semFlags,0FFh-semFlagOwnerKilled
	%Error(ercSemOwnerKilled)
	JMP		SHORT SwRet

SwsInvalidHandle:
	%Error(ercSemInvalidHandle)
	JMP		SHORT SwRet

SwrClearNow:
	XOR		AX,AX					;erc = ercOK

SwErcRet:
; AX has erc
	POPF							;ENABLE
SwRestoreDsRet:
	MOV		DS,DsSave
SwRet:
	LEAVE
	RET		6						;AX has erc
SemWait ENDP

SemWaitForClearSub PROC NEAR
;**********************************************************************
; On entry and return, FS:SI points to waitRec.
; Interrupts are disabled.
; Returns with timeout value in AX: if zero, then semaphore timed out.
; On return, waitRec has been deleted.
;**********************************************************************
; At this point there is a waitRec queued to the semaphore. 
; The next process that does a clear will cause our process to be
; unsuspended, so we need to suspend ourselves with interrupts disabled.
	PUSH	FS						;Save pointer to waitRec
	PUSH	SI
	PUSH	oPcbRun
	CALL	KSuspendProcess
; Continue here when another process calls KUnsuspendProcess.
; Free the waitRec.
	POP		SI						;Restore pointer to waitRec
	POP		FS
	%DeQueueSemWaitRec
	%DeallocSemWaitRec
	MOV		AX,semWaitTimeout
	RET		
SemWaitForClearSub ENDP

SemWakeupSub PROC NEAR
;**********************************************************************
; On entry and return: 
;	FS:SI points to semWaitRec
;	ES:BX points to semRec.
; Interrupts are disabled.
; Note that on return, the waitRec may have been deleted
; if another process ran.
;**********************************************************************
; Check to see if we have already woken up the waiting process.
	BTS		semWaitFlags,semWaitFlagWokeUp_bit
	JC		SwsRet					;We already woke up this process.
; Increment the use count so that the SemRec will not be deleted
; until we are finished with it.
	INC		semUseCount
	PUSH	ES						;Save pointer to SemRec
	PUSH	BX
	PUSH	FS						;Save pointer to WaitRec
	PUSH	SI
; Check to see how we should wake up the waiting process.
; If the waitRec is for a notify or a muxWait, then send a message.
	TEST	semWaitFlags,semWaitFlagNotify_mask+semWaitFlagMux_mask
	JZ		SwsUnsuspend
; Send a message to wake up the waiting process.
	PUSH	semWaitExch
	PUSH	semHandle
	CALL	SEND
; Note that the SEND could cause another process to run!
	POP		SI						;Restore pointer to WaitRec
	POP		FS
; Check to see if this waitRec is for a notify
	TEST	semWaitFlags,semWaitFlagNotify_mask
	JZ		SwsDecUseCount
; Delete the waitRec for this notify.
	%DeQueueSemWaitRec
	%DeallocSemWaitRec
	JMP		SHORT SwsDecUseCount

SwsUnsuspend:
; If the waitRec is not for a notify or a muxWait, then unsuspend the process.
	MOV		AX,semWaitTssId
	SUB		AX,8
	MOV		ES,AX					;ES = data pointer to TSS of waiter
	PUSH	ES:[tss_oPcb]			;oPcb of waiter
	CALL	KUnsuspendProcess		;(oPcb)
%IF(%CheckConsistency) THEN (
	OR		AX,AX
	JZ		SwsUnsuspendOK
	PUSH	AX
	CALL	Crash
SwsUnsuspendOK:
)FI
	POP		SI						;Restore pointer to WaitRec
	POP		FS
SwsDecUseCount:
	POP		BX						;Restore pointer to SemRec
	POP		ES
; Decrement the use count for this semaphore.
	DEC		semUseCount
%IF(%CheckConsistency) THEN (
	JNS		SwsUseCountOk
	%CallDebugger
SwsUseCountOk:
)FI
SwsRet:
	RET
SemWakeupSub ENDP

%IF(%CheckConsistency) THEN (
PUBLIC ValidateSemNodeFree
ValidateSemNodeFree PROC NEAR
;**********************************************************************
; ValidateSemNodeFree(fDealloc)
; If fDealloc, then oSemRec points to sem to deallocate, and it should 
; not already be on the free list
; Uses GS,DI,EAX
;**********************************************************************
fDealloc	EQU  WORD PTR [BP+6]
oSemRec		EQU  WORD PTR [BP+4]
cFree		EQU  WORD PTR [BP-2*1]
	ENTER	2*1,0
	PUSH	DX						;Save DX
	MOV		cFree,0
; Make sure semNode is not already free.
	LGS		DI,semAvail
VsTestLast:
	CMP		DI,OFFSET DGroup:semAvail
	JNE		VsNotLast
	MOV		AX,GS
	CMP		AX,DGroup
	JE		VsEndList
VsNotLast:
	INC		cFree
	CMP		fDealloc,0
	JE		VsNext					;Not deallocating
	CMP		DI,oSemRec
	JNE		VsNext
; SemRec is already in the free list
	%CallDebugger	
VsNext:
	LGS		DI,DWORD PTR GS:[DI]
	JMP		VsTestLast
VsEndList:
; Make sure we have the right number in the free list.
	MOV		AX,cFree
	CMP		AX,nSemNodeFree
	JE		VsCountOK
	%CallDebugger	
VsCountOK:
	POP		DX						;Restore DX
	LEAVE
	RET		4
ValidateSemNodeFree ENDP
)FI

%IF(%Log) THEN (
PUBLIC LogSem
LogSem PROC NEAR
;**********************************************************************
; LogSem(code)
; ES:BX points to SemRec
;	To look at the log:
;		semoLog points to the last entry
;		each previous entry is further down in memory.
;		We start at the end of the log and go backwards to make it
;		easier to look at the log in the debugger. 
;	Each entry in the log is a doubleword.  
;	The low order word has the TSS of the caller.
;	The high order word has a code:
;**********************************************************************
code	EQU  WORD PTR [BP+4]
	ENTER	2*1,0
	LEAVE
	RET		2
LogSem ENDP
)FI


SemUtilCode	ENDS
	END
