;******************************* MODULE HEADER *******************************;
;*                                                                           *;
;* FILE: RealNub.asm                                                         *;
;*                                                                           *;
;* MACHINE: B20        LANGUAGE: ASSEMBLER  OS: CTOS                         *;
;*                                                                           *;
;* DESCRIPTION:                                                              *;
;*                                                                           *;
;* Initialization for entering protected mode after the boot rom has loaded  *;
;* the operating system image.                                               *;
;*                                                                           *;
;*                                                                           *;
;* HISTORY:                                                                  *;
;*                                                                           *;
;* MM/DD/YY VVVV/MM PROGRAMMER   / DESCRIPTION OF CHANGE (Most recent first) *;
;*                                                                           *;
;* 06/30/93 0303.39 s. gavarre   / clear cpu breakpoint registers          	 *;
;* 04/07/93 0303.38 D. Renaud    / isPc changed fTryExtExpand; move        	 *;
;*                               / InitSg5000 before expansion             	 *;
;* 03/26/93 0303.37 J. McGinty   / repair test of isPc                     	 *;
;* 03/25/93 0303.36 J. McGinty   / don't expand to ext memory on Comarch	 *;
;* 03/03/93 0303.35 D. Renaud    / Decompression into extended memory on PC  *;
;* 02/01/93 0303.34 S. Gavarre   / Add comarch support                       *;
;* 11/13/92 0303.33 F. Wesner    / Remove all PowerPort stuff to InitOs.plm  *;
;* 08/04/92 0303.32 P. Johansson / For hard disk PC/AT boot, check CMOS to   *;
;*                                 differentiate IDE/ST-506 and SCSI         *;
;* 06/22/92 0303.31 D. Renaud    / Make sure SG5000 not mistaken as a PC     *;
;* 05/20/92 0303.30 F. Wesner    / Add conditional for NW; freeze if not PP  *;
;* 04/27/92 0303.29 D. Renaud    / Fix bug to enable cluster card serial id  *;
;* 04/20/92 0303.28 D. Renaud    / Update cluster card serial number addr    *;
;* 04/06/92 0303.27 F. Wesner    / Identify PCAT by cpuid as well as the date*;
;* 03/06/92 0303.26 D. Renaud    / On PC , get serialized id from ClusterCard*;
;* 02/26/92 0303.25 F. Wesner    / Sniff BIOS signature to id Portable       *;
;* 11/15/91 0303.24 D. Renaud    / On PC, copy osBuffer from saveArea        *;
;* 10/21/91 0303.23 D. Renaud    / On PC, init bootblock.saOsBuffer          *;
;* 10/15/91 0303.22 D. Renaud    / Recognize bootstrap signature on PC       *;
;* 08/02/91 0303.21 F. Wesner    / CTOS on PC - build boot block             *;
;* 04/04/91 0303.20 A. Thurber   / Let OS handle spurious default interrupts *;
;*                               / instead of doing it in InitSG5000.        *;
;* 03/25/91 0303.19 S. Gavarre   / Change SG5000 serial id format            *;
;*                                                                           *;
;* 12/14/90 0303.18 A. Thurber   / Identify SG5000 without using ROM or      *;
;*                                 BIOS calls.  Put bCpuVer, bCpuId and      *;
;*                                 serialized id on stack for OS.            *;
;* 10/17/90 0303.17 A. Thurber   / Assert all memory (0A$0000h to 0F$FFFFh)  *;
;*                                 as system RAM.                            *;
;* 10/02/90 0303.16 GWH          / Removed SGEN video hole.                  *;
;* 09/05/90 0303.15 S. Gavarre   / Get bootRom version and cpu type before   *;
;*                                 protected mode.                           *;
;* 07/03/90 0303.14 GWH          / Merge SuperGen support (Converted from    *;
;*                                 PS/2 )                                    *;
;*                                                                           *;
;* 09/21/89  JA SRP cycle lights erc120.                                     *;
;* 11/01/88  KH merge in PS2 support                                         *;
;* 09/20/88  JA move relbootrom to InitOs_all.plm.                           *;
;* 09/16/88  JA move relbootrom out of magic incantation.                    *;
;* 08/17/88  JM GP support.                                                  *;
;* 10/22/87  FW 386i support.                                                *;
;* 08/17/87  FW UserSignon (appended to sysErrorBuf) relocation also.        *;
;* 03/29/87  DR sysErrorBuf relocation                                       *;
;* 03/20/87  DR display sorry message for CP-001 or CWS, character mapped    *;
;* 02/11/87  DR relocate bootblock, xbus table, and iBus table if expansion  *;
;*              could overwrite them                                         *;
;* 12/08/86  DR image expansion support                                      *;
;* 10/11/86  DR NewGen support                                               *;
;* 04/11/85  DR revised                                                      *;
;*                                                                           *;
;* TITLE:  CTOS SYSTEM SOFTWARE                                              *;
;*                                                                           *;
;*            PROPRIETARY PROGRAM MATERIAL                                   *;
;*                                                                           *;
;* THIS MATERIAL IS PROPRIETARY TO UNISYS CORPORATION AND IS NOT TO BE       *;
;* REPRODUCED, USED OR DISCLOSED EXCEPT IN ACCORDANCE WITH PROGRAM LICENSE   *;
;* OR UPON WRITTEN AUTHORIZATION OF THE PATENT DIVISION OF UNISYS            *;
;* CORPORATION, DETROIT, MICHIGAN 48232, USA.                                *;
;*                                                                           *;
;* COPYRIGHT (C) 1980, 1990 CONVERGENT INCORPORATED. ALL RIGHTS RESERVED     *;
;* COPYRIGHT (C) 1980, 1990 UNISYS CORPORATION. ALL RIGHTS RESERVED          *;
;* ************************************************************************* *;
;*                                                                           *;
;* UNISYS BELIEVES THAT THE SOFTWARE FURNISHED HEREWITH IS ACCURATE AND      *;
;* RELIABLE, AND MUCH CARE HAS BEEN TAKEN IN ITS PREPARATION.  HOWEVER,      *;
;* NO RESPONSIBILITY, FINANCIAL OR OTHERWISE, CAN BE ACCEPTED FOR ANY        *;
;* CONSEQUENCES ARISING OUT OF THE USE OF THIS MATERIAL, INCLUDING LOSS OF   *;
;* PROFIT, INDIRECT, SPECIAL, OR CONSEQUENTIAL DAMAGES, THERE ARE NO         *;
;* WARRANTIES WHICH EXTEND BEYOND THE PROGRAM SPECIFICATION.                 *;
;*                                                                           *;
;* THE CUSTOMER SHOULD EXERCISE CARE TO ASSURE THAT USE OF THE SOFTWARE      *;
;* WILL BE IN FULL COMPLIANCE WITH LAWS, RULES AND REGULATIONS OF THE        *;
;* JURISDICTIONS WITH RESPECT TO WHICH IT IS USED.                           *;
;*                                                                           *;
;*                                                                           *;
;*************************** END OF MODULE HEADER ****************************;

%SET(DebugPC,0)
; If DebugVideo is set to 0FFh, real nub will display amount of memory
; in paragraphs required for decompression of the image (PC only).
%SET(DebugVideo,0)

$MOD286

$include(:f1:ioaddr1.edf)

trailerRaStart  equ word ptr es:[00]  ; raStart
trailerSaStart  equ word ptr es:[02]  ; saStart 
trailerRaStack  equ word ptr es:[04]  ; raStack
trailerSaStack  equ word ptr es:[06]  ; saStack
trailerDGdt     equ word ptr es:[08]  ; dGDT
trailerDGdtBaseLo equ word ptr es:[10]
trailerDGdtBaseHi equ byte ptr es:[12]
trailerDIdt     equ word ptr es:[16]  ; dIDT
trailerDIdtBaseHi equ byte ptr es:[20]
trailerSgDs     equ word ptr es:[24]
trailerSaSource equ word ptr es:[26]
trailerSaTarget equ word ptr es:[28]
trailerCWNub    equ word ptr es:[30]

cbOsBuffer      equ 36
cwOsBuffer      equ 18
cbBootBlock     equ 20
cwBootBlock     equ 10

cParTrailer     equ 2
cParBusTables   equ 20h
cParBootBlock   equ 2
cParCTOSData    equ 5 ; BootBlock, SysTime, SysErrorBuf, UserSignon

kbdControlRegPCAT  equ 64H ;   8042
kbdDataRegPCAT     equ 60H

maskNonSysSeg   equ 10h
rBaseLo equ 2
rBaseHi equ 4
rAccess equ 5
baseHi1Mb equ 10h

; bootBlock SCAT pointer and fields
raBootBlock         equ 1FCh
saBootBlock         equ 1FEh
raOsBuffer          equ 4
saOsBuffer          equ 6
;bWsDumpDevClass    equ 8
;bWsBootDevClass    equ 9
;bWsType            equ 10
;bWsDumpUnitNum     equ 11
;bWsBootUnitNum     equ 12
bWsRomVersion       equ 13

; Machines with CTOS value added BIOS
pBootBlockBIOS      equ 500h
lBiosSignatureLow   equ 'ot'
lBiosSignatureHigh  equ 'bo'
raSerialIdSG5000    equ 0		; P1, make sure these are right
saSerialIdSG5000    equ 0EFFFh

; CTOS Machines without BIOS
raRomVerId          equ 1FE6h
saRomVerId          equ 0FE00h
lSerializedNewGen   equ 11h
raSerialIdNewGen    equ 1FDEh
lSerializedNGen     equ 10h
raSerialIdNGen      equ 1FF8h

; PC/AT machine
lfirstSlsh      equ byte ptr es:[1FF7h]  ; first '/' in PC BIOS date
lscndSlsh       equ byte ptr es:[1FFAh]  ; second '/' in PC BIOS date
PCver           equ byte ptr es:[1FFEh]  ; PC model code
saSaveArea      equ 9F60h
;saPCFence       equ 1000h ; for debugging - so it will always happen
saPCFence       equ saSaveArea ; for production
magicT          equ 1234
saInt19         equ word ptr [66h]    ; sa of int 19h (bootstrap) handler
bClusterCardVer equ byte ptr es:[1Bh]
lVerCTOS        equ 2
raSerialIdClusterCard equ 3FB0h

; NGEN machines
RomDate         equ word ptr es:[1FEEh]  ; NGen ROM date

; Machine types
lSG5000             equ 0Fh
lPS2                equ 0F8h
lPCAT               equ 0FCh
lComarch            equ 020h

; Boot signatures - for recognition of PC boot sources - from the
; CTOS Hall of Fame
diskBootDI  equ 'FW'  ; Fred Wesner
diskBootSI  equ 'DR'  ; David Renaud
clstBootDI  equ 'GH'  ; Gary Hanning
clstBootSI  equ 'SN'  ; Shari Nolan
bootstrapDI equ 'JM'  ; John McGinty
bootstrapSI equ 'JF'  ; Jim Frandeen
comarchDI   equ 'SG'  ; Sherri Gavarre
comarchSI   equ 'JA'  ; Joe Altmaier

; Stack and Data are all in code segment for easy access
dgroup group bootseg 

bootseg segment word public 'code'
assume cs:bootseg, ds:nothing, es:nothing

bootBlock           dd 0
bootBlockraOsBuffer dw offset osBuffer
bootBlocksaOsBuffer dw 0
bWsDumpDevClass     db 0
bWsBootDevClass     db 0
bWsType             db 0
bWsDumpUnitNum      db 0
bWsBootUnitNum      db 0
bootBlockRomVersion db 0
                    dw 0
                    dw 0
                    dw 0

; the OSBuffer MUST follow the boot block (expansion depends on it)
osBuffer  dw 0      ; sysTime
          dw 0
          dw 0
          dw 1234   ; magicT

sysErrorBufStatus  dw 0
sysErrorBufProcess dw 0
          dw 0
          dw 0
          dw 0
          dw 0
          dw 0
          dw 0

          db 12 dup (0)  ; signon

; end of OsBuffer

%IF (%debugpc) THEN (
;************ASSERT DEBUG STUFF**************
vidbuff dw 0b800h
cursor  dw 0

FBootPath label word
    db 19,'Booted from floppy!'
HBootPath label word
    db 18,'Booted from hdisk!'
CBootPath label word
    db 20,'Booted from cluster!'
;***********END*ASSERT DEBUG STUFF***********
)FI



     dw 80 dup (0)      ; Need minimum of 36 bytes for osBuffer
StackTop label word
     dw 5A5Ah           ; So stack looks normal when debugging crash dumps

;--------  GDT for expanding image into extended memory ------
; 0 - use as gdtr
gdtr	dw 17h       ; limit
gdt_lo	dw 0
gdt_hi	db 0
		db 0
		dw 0
; 8 - huge identity selector
		dw 0FFFFh    ; limit
		dw 0000
		dw 9200h
		dw 00CFh
; 10 - identity descriptor for real mode reentry
		dw 0FFFFh    ; limit
		dw 0000
		dw 9200h
		dw 0000

fTryExtExpand db 0
serialIdAx dw 0
serialIdDx dw 0

; ***** Entry point *****

;**************************************************************************
;
; RealNub entry:
;
; On Unisys non-PC machines (B3x and SG-xxxx Series) with special firmware
; that builds a CTOS boot block, the value in DX is NOT USED. 
;
; On PC style machines DX is set up by the boot record before it calls us -
;
;    IF bootstrap signature then
;       bootblock was set up by Bootstrap, so do nothing
;
;    ELSE IF disk signature then 
;       DO; /* CTOS was loaded by a Unisys disk boot record */
;           /* and the boot device is in DL as follows:     */
;       IF DL = 0 then booted from floppy unit 0 (or A:)
;       ELSE IF DL = 1 then booted from floppy unit 1 (or B:)
;       ELSE IF DL = 80H then booted from hard disk unit 0 (or C:)
;       ELSE IF DL = 81H then booted from hard disk unit 0 (or D:)
;       END;
;
;    ELSE IF cluster card signature then 
;       DO; /* Booted by new cluster card */
;       wsType = DX; /* New cluster card provides wsType*/
;       END;
;
;    ELSE DO; /* Booted by old cluster card - use fixed wsType */
;       wsType = 202;  /* force wstype to 202 */
;       END;
;
;
;
;**************************************************************************

public RealNub
RealNub proc far

	cli

	mov  bx, di       ;save signature 1
	mov  cx, si       ;save signature 2
	mov  ax, cs
	mov  ss, ax
	mov  sp, offset StackTop

	push cx     ;signature 2
	push bx     ;signature 1
	push dx     ;save any passed parameter from boot record - see comment above

; verify that processor supports protected mode
; only 8086/80186 have 1 in bit 15 of the flags register.
	pushf
	pop  dx 
	shl  dx, 1
	jnc  TestForBIOS
	call SorryCantGoProtected	; never returns

TestForBIOS:

; Clear the CPU breakpoint registers 

$MOD386
	xor  eax, eax
	mov  Dr0, eax
	mov  Dr1, eax
	mov  Dr2, eax
	mov  Dr3, eax
	mov  Dr7, eax
$MOD286

; Look for a SG-5000 BIOS created bootBlock via pointer at 500h.
; SG-5000 BIOS can't put pointer at 1FCh because of DOS IDT conflict.
	xor  ax, ax
	mov  ds, ax
	les  bx, dword ptr [pBootBlockBIOS]
	cmp  word ptr es:[bx], lBiosSignatureLow
	jne  TestForPC              ;
	cmp  word ptr es:[bx+2], lBiosSignatureHigh
	jne  TestForPC              ;
	add  bx, 4                  ; signature is not part of bootBlock
	mov  [raBootBlock], bx      ;
	mov  [saBootBlock], es      ;
ItsAnSG5K:
	call InitSg5000
TestForExpansionRly:            ;target for short jumps
	pop  dx                     ;pop DL from bootblock- not needed for SG-5000
	pop  dx                     ;pop unneeded signature
	pop  dx                     ;pop unneeded signature
	jmp  TestForExpansion       ;

TestForPC:
; Make sure a SG5000 does not get mistaken as a PC
	les  bx, dword ptr [raBootBlock]
	mov  ah, byte ptr es:[bx+bWsRomVersion]	; bCpuId on BIOS machines
	cmp  ah, lSG5000
	je   ItsAnSG5K

	mov  cx,0FE00h              ;
	mov  es,cx                  ;

	mov  al,lfirstSlsh          ;
	mov  ah,lscndSlsh           ;
	cmp  ax,'//'                ;
	je   PassedPcInitialTest    ;jif if smells like a PC
	                            ;Might not be a PC.  See if it resembles an
	                            ;NGen by checking the date in the rom.
	mov  ax, RomDate            ;get the date
	cmp  ax, 0081h              ;check against 1981
	jle  TestForPC2             ;wrong date, it might be a PC
	cmp  ax, 0099h              ;check against 1999
	jg   TestForPC2             ;wrong date, it might be a PC
; at this point the machine smells like an NGen
	jmp  short TestForExpansionRly    ;Not a PC
TestForPC2:
PassedPcInitialTest:
	mov  al, PCver              ;get version byte
	cmp  al, lPCAT              ;is this an AT?
	je   ItsAnAT                ;yes
	cmp  al, lPS2               ;is this a PS/2?
	jne  TestForExpansionRly    ;Not an AT or PS/2; treat as an NGen.
ItsAnAt:
	mov  fTryExtExpand, 0FFh
	mov  bootBlockRomVersion,al ;store PC type byte in bootblock
%if (%DebugVideo) then (
	mov ax, 0003h  ; set video mode to mode 3 (ah = 0)
	int 10h

; put cursor at row 24, column 0
	mov ax, 0200h
	xor bx, bx
	mov dx, 1800h
	int 10h
) fi

	pop  dx                     ;DL has boot parameter from bootrecord
	pop  di                     ;saved signature 1
	pop  si                     ;saved signature 2

; determine where the boot came from using the signatures

; determine where the boot came from using the signatures
	cmp  di, comarchDI
	jne  notComarch
	cmp  si, comarchSI
	jne  notComarch
	jmp  TestForExpansion	; bootblock was set up by rom
notComarch:
	cmp  di, bootstrapDI
	jne  notBootstrap
	cmp  si, bootstrapSI
	jne  notBootstrap
	jmp  TestForExpansion	; bootblock was set up by Bootstrap
notBootstrap:
	mov  bx, offset BootBlock
	mov  ds:[raBootBlock], bx   ;set up SCAT pointer
	mov  ds:[saBootBlock], cs   ;ditto
	cmp  di, diskBootDI         ;check for disk boot signature
	jne  notPCdiskBoot          ;jif no match
	cmp  si, diskBootSI         ;check second word of signature
	jne  notPCdiskBoot          ;jif not disk boot

; Called by disk boot sector    ;
	mov  dh, 6                  ;NEC floppy romClass
	test dl, 80h                ;Booted from floppy?
	jz   WsType                 ;Yes
	and  dl, 7Fh                ;Keep only the unit number
	push dx                     ;Save across ReadCmos call
	push 12h                    ;CMOS byte 12h describes C: and D:
	call ReadCmos
	pop  dx
	mov  dh, 2                  ;ST-506/IDE romClass
	or   al, al                 ;Is there an ST-506/IDE present?
	jnz  WsType                 ;Yes, leave DH as is
	mov  dh, 5                  ;SCSI romClass
WsType:	xor  cx, cx                 ;sets wsType to zero
	jmp  short PCcontinue       ;DH = disk device type, DL = disk unit number

; Called by cluster card boot rom
notPCdiskBoot:                  ;
	cmp  di, clstBootDI       ;check for new cluster card boot signature
	jne  tryOldClstCard         ;jif no match
	cmp  si, clstBootSI       ;check second word of signature
	jne  tryOldClstCard         ;jif not disk boot
	mov  cx, dx                 ;CL = passed wsType
	jmp  short setWsType        ;

tryOldClstCard:
	mov  cl, 202                ;default boot device type
setWsType:
	xor  dl,dl                  ;set station num to zero, we don't know it
	mov  dh, 3                  ;device type - booting from cluster

PCcontinue:
	mov  bWsBootDevClass, dh    ;boot device type
	mov  bWsBootUnitNum, dl     ;save unit num
	mov  bWsType, cl            ;number of OS that was T-booted.
	mov  bWsDumpDevClass, dh    ;set dump type = boot type
	mov  bWsDumpUnitNum, dl     ;set dump num = boot num

	mov bootBlockSaOsBuffer, cs

; If it is valid, build OsBuffer from saveArea

	push ds
	mov ax, saSaveArea
	mov ds, ax
	mov ax, word ptr [2]
	add ax, word ptr [4]
	add ax, word ptr [6]
	cmp ax, magicT
	jne @@1

	push cs
	pop es
	mov si, 2
	mov di, offset OsBuffer + 2
	mov cx, 11
	cld
	repnz movsw
@@1:
	pop ds

%if (0) then (
	push RTCmosCrashErcAddr1
	call ReadCmos
	xchg ah, al
	push RTCmosCrashErcAddr0
	call ReadCmos
	mov sysErrorBufStatus, ax
	or ax, ax
	jz @1

; crash happened - build OS buffer and clear CMOS
	push RTCmosCrashProcAddr
	call ReadCmos
	xor ah, ah
	mov sysErrorBufProcess, ax 


 	push RTCmosCrashErcAddr0
	push 0
	call WriteCmos

 	push RTCmosCrashErcAddr1
	push 0
	call WriteCmos

@1:
)FI

%IF (%debugpc) THEN (
;*****Assert DEBUG******REMOVE THIS STUFF LATER******
	cmp   dh,0h
	je    bfhd
	cmp   dh,06h
	je    bffd
	push  cs
	pop   ds
	mov   si,offset CBootPath
	mov   cursor,0
	call  DoMessage
	jmp short endmsg
bffd:
	push  cs
	pop   ds
	mov   si,offset FBootPath
	mov   cursor,0
	call  DoMessage
	jmp short endmsg
bfhd:
	push  cs
	pop   ds
	mov   si,offset HBootPath
	mov   cursor,0
	call  DoMessage
endmsg:
	mov cx,150
outerloop:
	mov bx, cx
    xor cx,cx
	loop $
	mov cx,bx
	loop outerloop

;*****Unassert DEBUG
)FI

TestForExpansion:
	mov  ax, cs
	sub  ax, cParTrailer
	mov  es, ax				; es addresses trailer
	mov  cx, trailerCwNub
	cmp  cx, 0FFFFh			; if trailerCwNub <> 0FFFFh then do expansion
	jne  RelocateData
	jmp  ExpansionDone

RelocateData:
; Relocation is necessary to guarantee that expansion of the OS does not 
; overwrite data areas.  Relocate the IBus table, XBus table, bootBlock and 
; osBuffer to the end of the relocated RealNub.  The boot block pointer at 
; 1FCh addresses the bootBlock.  The IBus table preceeds by 512 bytes.  The 
; XBus table preceeds by 256 bytes.  pOsBuffer gets fixed up after osBuffer is 
; moved.  When done we will have:
;       Nub         -  ? paragraphs (relocation table, trailer, RealNub)
;       IBus table  - 16 paragraphs
;       XBus table  - 16 paragraphs
;       Boot Block  -  2 paragraphs
;       SysTime     -  8 bytes
;       SysErrorBuf - 16 bytes
;       UserSignon  - 12 bytes

; bx = saTarget, dx = saBootBlockNew, cx = cwNub from above
	add  cx, 7					;round cwNub up to count of paragraphs
	shr  cx, 3
	mov  bx, trailerSaTarget	;start of nub (relocation table) after expand
	add  bx, cx					;bx points to end of RealNub after expansion
	mov  dx, bx
	add  dx, cParBusTables		;dx points to end of tables, start of bootblock

; Check to see if decompression would go exceed useable
; memory in the 640k space.  If so, do decompression into extended memory.

	test  fTryExtExpand, 1 	
	jz ContinueRelocateData
	
	mov ax, saPCFence
	sub ax, cParCTOSData

%if (%DebugVideo) then (
	push ax
	mov ax, dx
	add ax, cParBusTables
	call DisplayAx
	pop ax
) fi

	cmp ax, dx
	ja ContinueRelocateData

; don't expand to extended memory on the Comarch, it has no 640k hole, has
; no 8042 controller.
	push es	
	push 0
	pop  es
	les  di, es:dword ptr [raBootBlock]
	cmp  byte ptr es:[di+bWsRomVersion], lComarch	; bCpuId on BIOS machines
	pop	 es ; restore saTrailer
	je   ContinueRelocateData

	jmp ExpandToExtendedMemory


; Save osBuffer on stack before relocating tables and bootBlock
ContinueRelocateData:
	xor  di, di
	mov  es, di
	lds  si, dword ptr es:[raBootBlock]	; pBootBlock
	lds  si, dword ptr [si+raOsBuffer]	; pOsBuffer
	sub  sp, cbOsBuffer
	mov  di, sp
	push ss
	pop  es
	mov  cx, cwOsBuffer
	cld
	repnz movsw

; fetch pBootBlockOld (ds:si) again, zero di
	xor  di, di
	mov  es, di
	lds  si, dword ptr es:[raBootBlock]

; Fix SCAT to point to new bootBlock
	mov  es:[raBootBlock], di	; must be zero for OS
	mov  es:[saBootBlock], dx

; Fix pOsBuffer in old bootBlock before we move it
	add  dx, cParBootBlock		; osBuffer follows bootBlock
	mov  [si+raOsBuffer], di
	mov  [si+saOsBuffer], dx

; ds,ax = saSource, ds:si = pSource
	mov  ax, ds
	sub  ax, cParBusTables
	mov  ds, ax
; es,bx = saTarget, es:di = pTarget
	mov  es, bx

; If pSource < pTarget then do reverse move.
;
; Note: pSource may have non-zero offset, pTarget has zero offset, so
; need to normalize pSource. Normalized offset of pSource is irrelevant
; to the comparison because we are dealing in paragraphs.

	push dx			; save saOsBufferRelocated
	mov  dx, si 
	shr  dx, 4
	add  ax, dx
	pop  dx

	mov  cx, cParBusTables+cParBootBlock
	shl  cx, 3		; cwMove

; ax = normalized saSource, bx = saTarget
	cmp  ax, bx
	jb   RelocateBootR
	cld
	repnz movsw
	jmp  RelocateOsBuffer
RelocateBootR:
	mov  ax, cx		; calculate byte offset from cwMove
	dec  ax
	shl  ax, 1
	add  si, ax
	add  di, ax
	std
	repnz movsw

RelocateOsBuffer:
	push ss			; saved osBuffer is at ss:sp
	pop  ds
	mov  si, sp
	mov  es, dx		; dx is saOsBufferRelocated
	xor  di, di
	cld
	mov  cx, cwOsBuffer
	repnz movsw
	add  sp, cbOsBuffer

; Relocate the nub so that it is out of the way for expansion
	mov  ax, cs
	sub  ax, cParTrailer
	mov  es, ax  ; es addresses trailer
	mov  cx, trailerCwNub
	mov  ax, trailerSaSource
	mov  ds, ax
	xor  si, si
	mov  dx, trailerSaTarget
	mov  es, dx
	xor  di, di
	cld
	repnz movsw

; Setup for the relocated code and jump to it
	sub  dx, ax 	; dx is cParDifferent, i.e. how far we moved the nub
	mov  ax, cs
	add  ax, dx 	; csNew = csOld + cParDifferent
	mov  ss, ax 	; set the new stack
	mov  sp, offset StackTop
	push ax			; cs
	push offset ExpandImage
	ret

ExpandImage:		; Now running relocated RealNub on relocated stack
	sub  ax, cParTrailer
	mov  es, ax		; es addresses relocated trailer
	mov  ax, trailerSaTarget
	mov  ds, ax		; ds:si addresses relocation structure
	xor  si, si

ExpandLoop:
	cld
	lodsw
	mov  cx, ax		; 0=done, 1=expand as zeroes, 2=relocate data
	jcxz ExpansionDoneRelay
	dec  cx
	jcxz ZeroCase
	dec  cx
	jcxz RelocateCase
	jmp .+0			; The table is bogus

RelocateCase:
	lodsw
	mov  dx, ax		; saSource
	lodsw
	mov  es, ax		; saTarget
	lodsw
	mov  cx, ax		; cwRelocate
	mov  di, cx		; The table is such that we start in high memory and work
	dec  di			; down to the bottom.  Get byte offset from cwRelocate.
	shl  di, 1
	push ds			; Save current relocation table pointer
	push si
	mov  ds, dx
	mov  si, di		; Paragraph aligned, source ra = target ra
	std
	repnz movsw
	pop  si
	pop  ds
	jmp  ExpandLoop

ZeroCase:
	lodsw
	mov  es, ax		; saTarget
	lodsw
	mov  cx, ax		; cwZeroes
	xor  di, di		; Paragraph aligned, ra=0
	xor  ax, ax		; storing zeroes
	cld
	repnz stosw
	jmp  ExpandLoop
ExpansionDoneRelay:
	jmp ExpansionDone

ExpandToExtendedMemory:
$MOD386
; ASSERT - es addresses trailer
; Poke the 8042 keyboard controller to turn off 1Mb wrap; Gate A20 
	call WaitControllerReady
; cmnd write output port */
	mov dx, kbdControlRegPCAT
	mov ax, 0D1h
	out dx, al
	call WaitControllerReady
; Enable A20 
	mov dx, kbdDataRegPCAT
	mov ax, 0DFh
	out dx, al
	call WaitControllerReady

; add 1Mb relocation to gdt and idt base
	or trailerDGdtBaseHi, baseHi1Mb
	or trailerDIdtBaseHi, baseHi1Mb
	mov  ax, trailerSaSource
	mov  ds, ax		; ds:bx addresses relocation structure
	xor  bx, bx

; initialize real nub gdt base
	xor eax, eax
	mov ax, cs
	shl eax, 4
	add eax, offset gdtr
	mov gdt_lo, ax
	shr eax, 16
	mov gdt_hi, al

ExtMemExpandLoop:
	xor esi, esi    ; zeroes source and sets op indicator for zero case
	mov cx, [bx]	; 0=done, 1=expand as zeroes, 2=relocate data
	add bx, 2
	jcxz FixupGdt
	dec  cx
	jcxz ExtMemZeroCase
	dec  cx
	jcxz ExtMemCopyCase
	jmp .+0			; The table is bogus

ExtMemCopyCase:
	mov si, [bx]
	add bx, 2
	shl esi, 4		; o32Source
	or si, 1        ; sets op indicator for copy case

ExtMemZeroCase:
	xor edi, edi
	mov di, 1       ; add in 1Mb relocation 
	shl edi, 16
	mov di, [bx]
	add bx, 2
	shl edi, 4      ; o32Target

	xor ecx, ecx
	mov cx, [bx]    ; cwRelocate
	shr cx, 1       ; use dword count
	add bx, 2

; go protected - esi, edi, ecx setup for copy or zeroing
	push ds
	lgdt gdtr				; load the gdt base and limit
	smsw ax					; set protected mode bit
	or ax, 1
	lmsw ax 
	jmp .+2					; clear bogus pre-fetch queue

; ASSERT - in protected mode
; CS and SS reference same addresses as before protected switch

	mov ax, 8 ; selector 8 is the huge zero-based selector
	mov ds, ax
	mov es, ax
	cld

	mov ax, si
	and si, 0FFFEh ; clear the op indicator
	rcr ax, 1
	jnc DoZeroOp
; copy operation
	db 66h
	db 67h
	db 0F3h ; rep
	db 0A5h ; movsd
	jmp short GoReal

DoZeroOp:
	xor eax, eax
	db 66h
	db 67h
	db 0F3h ; rep
	db 0ABh ; stosd

GoReal:
	mov ax, 10h; reload cache for segment regs that were used
	mov es, ax
	mov ds, ax

; set PE bit to zero
	mov eax, cr0
	and  al, 0FEh
   	mov cr0, eax
	jmp .+2

; ASSERT - back in real mode
; CS and SS reference same addresses as before protected switch

	pop ds
	jmp  ExtMemExpandLoop

FixupGdt:
; get gdt base in ebx
	mov ax, cs
	sub ax, cParTrailer
	mov es, ax				; es addresses trailer
	xor ebx, ebx
	mov bl, trailerDGdtBaseHi
	shl ebx, 16
	mov bx, trailerDGdtBaseLo
; get count of selectors in cx
	mov cx, trailerDGdt
	inc cx
	shr cx, 3
; go protected again - to add 1Mb relocation to gdt selectors
	lgdt gdtr				; load the gdt base and limit
	smsw ax					; set protected mode bit
	or ax, 1
	lmsw ax 
	jmp .+2					; clear bogus pre-fetch queue

; ASSERT - in protected mode
	mov ax, 8 ; selector 8 is the huge zero-based selector
	mov ds, ax
GdtFixupLoop:
; skip if not code or data
	mov al, byte ptr [ebx+rAccess]
	test al, maskNonSysSeg
	jz NextSelector
; skip if base is zero
	mov ax, word ptr [ebx+rBaseLo]
	or  al, byte ptr [ebx+rBaseHi]
	or ax, ax
	jz NextSelector

	or byte ptr [ebx+rBaseHi], baseHi1Mb
NextSelector:
	add ebx, 8
	loop GdtFixupLoop

	mov ax, 10h; reload cache for segment regs that were used
	mov ds, ax

; set PE bit to zero
	mov eax, cr0
	and  al, 0FEh
   	mov cr0, eax
	jmp .+2
; ASSERT - in real mode, again

ExpansionDone:
$MOD286
;Get the bootrom version, cpuId and serialized id while in real mode.
;On SGen machines bCpuId is in the bootBlock.  Fetch it from ROM on all others.

	xor  ax, ax
	mov  ds, ax
	les  bx, dword ptr [raBootBlock]
	mov  ah, byte ptr es:[bx+bWsRomVersion]	; bCpuId on BIOS machines
	cmp  ah, lSG5000
	je   GetSG500Ver
	cmp  ah, lComarch		; bCpuVer is 32 
	jne  IsitPC
	push ax                 ; bCpuVer
	mov  ax, saRomVerId
	mov  es, ax
	mov  bx, raSerialIdNGen
	jmp  GetSerialId

IsitPC:
	cmp  ah, lPS2
	jne  NotPS2
	mov  ah, 17             ; bCpuVer is 17 for PS/2
	jmp  short SetPCver     ;
NotPS2:
	cmp  ah, lPCAT
	jne  GetIdFromROM       ; jif NGen
	mov  ah, 16             ; bCpuVer is 16 for PCAT

SetPCVer:
	xor  al, al             ; no bRomVer
	push ax                 ;

	mov ax, saInt19         ; if CTOS cluster card is present, get serial id
	cmp ax, 0F000h
	jae NoSerialId          ; cluster card must be present

	mov es, ax              ; es = cluster card base address
	mov al, bClusterCardVer
	cmp al, lVerCTOS
	jne NoSerialId

	mov  bx, raSerialIdClusterCard
	mov  dh, es:[bx]		; Reverse the order of the bytes
	mov  dl, es:[bx+1]
	mov  ah, es:[bx+2]
	mov  al, es:[bx+3]

	jmp  short PushSerialId

GetSG500Ver:
	mov  al, 0				; bRomVer unused on SG5000, all are serialized
	push ax					; Push bRomVer,bCpuId now
	mov ax, serialIdAx
	mov dx, serialIdDx
	jmp  short PushSerialId

GetIdFromROM:
	mov  ax, saRomVerId
	mov  es, ax
	mov  bx, raRomVerId
	mov  ax, es:[bx]
	push ax					; Push bRomVer,bCpuId for OS
	mov  bx, raSerialIdNGen
	cmp  al, lSerializedNGen
	je   GetSerialId
	mov  bx, raSerialIdNewGen
	cmp  al, lSerializedNewGen
	je   GetSerialId
NoSerialId:
	xor  ax, ax				; Not a serialized bootrom
	mov  dx, ax
	jmp  short PushSerialId
GetSerialId:
	mov  dh, es:[bx]		; Reverse the order of the bytes
	mov  dl, es:[bx+2]
	mov  ah, es:[bx+4]
	mov  al, es:[bx+6]

PushSerialId:
	push dx					; Push serialized id for OS
	push ax

; Load initial cs:ip and descriptor table registers from trailer
	mov  ax, cs
	mov  ds, ax
assume ds:dgroup
	sub  ax, cParTrailer
	mov  es, ax				; es addresses trailer
	mov  di, trailerRaStart	; initial ip
	mov  si, trailerSaStart	; initial cs

; Switch to protected mode
	lgdt trailerdGDT				; load the gdt base and limit
	lidt trailerdIDT				; load the idt base and limit

	smsw ax					; set protected mode bit
	or ax, 1
	lmsw ax 
	jmp  .+2				; clear bogus pre-fetch queue

; Until loaded, cs and ss reference same addresses as before protected switch
; Setup OS entry cs:ip on stack for RET below
	push si					; cs
	push di					; ip

; Initialize a few things for the OS, not really necessary
	xor  ax, ax
	mov  ds, ax
	mov  es, ax
	mov  bx, ax
	mov  cx, ax
	mov  dx, ax
	mov  si, ax
	mov  di, ax
	mov  bp, offset StackTop	; gratuitous, for debugging dumps only
	cld

; stack now looks like:
; sp->  ip
;       cs
;       serialIdLow
;       serialIdHigh
;       bRomVer,bCpuId
; bp->  5A5Ah

	ret						; Jump to the OS

RealNub endp



; ***** Utility subroutines *****

lShadowRomEnableReg equ 0C02h
lSramMapControlReg  equ 0C03h

InitSG5000 proc near
assume cs:bootseg, ds:nothing, es:nothing

; Initialize all address ranges to known state

; Address range         Boot Rom state (?)          Desired OS init state
; -------------         --------------              ---------------------
; 00$0000h to 07$FFFFh  System RAM                  System RAM
; 08$0000h to 09$FFFFh  System RAM                  System RAM
; 0A$0000h to 0A$FFFFh  Off-board Video RAM         System RAM
; 0B$0000h to 0B$FFFFh  Off-board Video RAM         System RAM
; 0C$0000h to 0C$FFFFh  Off-board ROM, shadowed     System RAM, shadowed
; 0D$0000h to 0D$FFFFh  Off-board ROM               System RAM
; 0E$0000h to 0F$FFFFh  System RAM, shadowed        System RAM, shadowed

; NOTE: SGV-100 Video RAM will respond to 08$0000h-0B$FFFFh
;       if those addresses are allowed onto the X-Bus+

; Make sure BIOS ROM is enabled (not shadowed) for fetch of serialized id
	mov  dx, lShadowRomEnableReg
	mov  al, 2					; ExRomEn#=1, RomEn#=0
	out  dx, al
	jmp  .+2

; Get the serialized id from the ROM
	mov  ax, saSerialIdSG5000
	mov  es, ax
	mov  bx, raSerialIdSG5000
	mov  dh, es:[bx]			; Reverse the order of the bytes
	mov  dl, es:[bx+1]			; bytes are sequential since bios 
	mov  ah, es:[bx+2]			; is only one part
	mov  al, es:[bx+3]
	mov serialIdAx, ax
	mov serialIdDx, dx

; Make sure 0C$0000h to 0C$FFFFh and 0E$0000h to 0F$FFFFh are shadowed,
; i.e. RAM will respond to these addresses (provided SRAM is also correct)
	mov  dx, lShadowRomEnableReg
	mov  al, 3h					; ExRomEn#=1, RomEn#=1
	out  dx, al
	jmp  .+2

; Make 0A$0000h to 0F$FFFFh System RAM
	mov  bl, 04h				; no write-protect, cache enabled, on-board
	mov  ax, 0A000h
	call MapRamSegment
	mov  ax, 0B000h
	call MapRamSegment
	mov  ax, 0C000h
	call MapRamSegment
	mov  ax, 0D000h
	call MapRamSegment
	mov  ax, 0E000h
	call MapRamSegment
	mov  ax, 0F000h
	call MapRamSegment
	ret
InitSG5000 endp


MapRamSegment proc near
; On entry, ax is pla, bl is control byte
	mov  es, ax
	mov  dx, lSramMapControlReg
	mov  al, 02h				; SRAM enable
	jmp  .+2					; Clear pre-fetch queue
	out  dx, al					; Enable SRAM mapper to be programmed
	mov  byte ptr es:[0], bl
	mov  al, 04h				; SRAM disable
	out  dx, al					; Disable SRAM Mapper from being programmed.
	ret							; will clear pre-fetch queue, which is invalid
MapRamSegment endp


fOsInitialized      equ byte ptr es:[802h+34]	; cdt in GP OS image
myLedPort           equ 58h						; led port on 186 MF
vidControlT1        equ 0FA82h


SorryCantGoProtected proc near

; Turn off fOsInitialized in Cdt
	Xor	Ax, Ax
	Mov	Es, Ax
	Mov	fOsInitialized, Al 

%*DEFINE(Delay x)(
%Repeat(%x)
(	Loop .
))

%*DEFINE(Digit x)(
	Mov	Al, %Eval((NOT %x AND NOT 80h) AND 0FFh)
	Out	myLedPort, Al
	%Delay 10
	Mov	Al, 0FFh
	Out	myLedPort, Al
	%Delay 2
)

DoForever:
; Cycle all leds on but one.
	Mov	Al, 01h
	Mov	Ah, 32
LedCycle:
	Out myLedPort, Al		; Al is complement of lighted leds
	%Delay 1
	Ror	Al, 1
	Dec	Ah
	Jnz	LedCycle

; Blank for a bit
	Mov	Al, 0FFh
	Out	myLedPort, Al
	%Delay 5

; Display each hex digit of erc 120 = 0078h.
	%Digit 0
	%Digit 0
	%Digit 7
	%Digit 8

	Jmp	DoForever


SorryCantGoProtected endp

WaitControllerReady proc near
	mov dx, kbdControlRegPCAT
@WCR:
	in al, dx
	test al, 02h
	jnz @WCR
	ret
WaitControllerReady endp

%IF (%debugpc) THEN (
;*****Assert DEBUG
DoMessage:
	push   ds           ;save it
	push   es
	push   ax
	push   cx
	push   di
	push   cs
	pop    ds
	mov    di,cursor
	mov    es,vidbuff
	xor    cx,cx
	mov    cl,byte ptr [si]  ;size of banner
	inc    si
	mov    ah,087h            ;attribute, white on black blinking
aloop:
	mov    al,byte ptr [si]
	mov    es:[di],ax
	inc    di
	inc    di
	inc    si
	loop   aloop
	add    cursor,160
	cmp    cursor,3840	
	jne    restoreregs
	mov    cursor,0
restoreregs:
	pop    di
	pop    cx
	pop    ax
	pop    es
	pop    ds
	ret

;*****Unassert DEBUG
)FI


%if (%debugVideo) then (

OutMark proc near
	pusha
	xor bx, bx      ;
 	mov ax, 0E40h
	int 10h
	popa
	ret
OutMark endp

DisplayAX proc near
	pusha
	mov cx, 4
	mov dx, ax
DisplayNext:
	rol dx, 4
	mov ax,dx
	and ax, 0fh
	cmp ax, 0Ah
	jae DisplayHexDigit
	add al, 30h
	jmp short DoDisplay
DisplayHexDigit:
	add al, 37h
DoDisplay:
	push dx
	mov ah, 0Eh
	int 10h         ; video
	pop dx
	loop DisplayNext
	mov ax, 0E2fh
	int 10H
	popa
	ret
DisplayAX endp


)fi

ReadCmos proc near ; (addr)
	push bp
	mov bp, sp
	mov dx, RTCmosAddressReg
	mov al, [bp+4]
	out dx, al
	jmp .+2
	mov dx, RTCmosDataReg
	in  al, dx
	pop bp
	ret 2

ReadCmos endp

%if (0) then (

WriteCMOS proc near ; (addr, data)
	push bp
	mov bp, sp
	mov dx, RTCmosAddressReg
	mov al, [bp+6]
	out dx, al
	jmp .+2
	mov dx, RTCmosDataReg
	mov al, [bp+4]
	out dx, al
	pop bp
	ret 4
WriteCMOS endp

)fi

bootseg ends

stack segment stack 'stack' ; prevents linker error
stack ends

end RealNub
